/* Copyright 2013 Humboldt University of Berlin
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program. If not, see <http://www.gnu.org/licenses/>.
*
* Author: Mihal Brumbulli <mbrumbulli@gmail.com>
*/

#include "demoddix.h"


int           ProgressWindow::width;
int           ProgressWindow::height;
int           ProgressWindow::xPos;
int           ProgressWindow::yPos;

std::string   ProgressWindow::message       = "";
unsigned char ProgressWindow::alpha         = 255;

// Create window
void ProgressWindow::Create() 
  {
  ProgressWindow::width = RootWindow::width - Window::STATE_W - 3 * Window::PADDING;
  ProgressWindow::height = Window::PROGRESS_H;
  ProgressWindow::xPos = Window::STATE_W + 2 * Window::PADDING;
  ProgressWindow::yPos = Window::PADDING;
  }

// Draw progress bar
void ProgressWindow::Display() 
  {
  // Viewport and crop
  glViewport(ProgressWindow::xPos, ProgressWindow::yPos, ProgressWindow::width, ProgressWindow::height);
  glScissor(ProgressWindow::xPos, ProgressWindow::yPos, ProgressWindow::width, ProgressWindow::height);
  glEnable(GL_SCISSOR_TEST);
  glLoadIdentity();
  glLineWidth(1.0f);
  // Ratio for mapping pixels to coordinates
  double xRatio = 2.0 / ProgressWindow::width;
  double yRatio = 2.0 / ProgressWindow::height;
  // Times in ms
  unsigned long long now = Demoddix::currentTime / 1000000; // in ms
  unsigned long long end = Demoddix::endTime / 1000000; // in ms
  unsigned long long step = Demoddix::stepTime / 1000000; // in ms
  // Progress bar value and text
  double progress = -1.0;
  char str[256];
  if (Demoddix::simulatorStatus() == Demoddix::OK && Demoddix::displayStatus() == Demoddix::OK)
    {
    sprintf(str, "time = %lld | step = %lld", now, step);
    ProgressWindow::message = std::string(str);
    progress += (2.0 * now) / end;
    }
  // Draw progress
  if (progress > -1.0)
    {
    glColor3ub(Window::BD_COLOR[0], Window::BD_COLOR[1], Window::BD_COLOR[2]);
    glRectd(-1.0, 1.0, progress, -1.0);
    }
  // Draw text
  double w = glutBitmapLength(Window::FONT, (const unsigned char*) ProgressWindow::message.c_str()) * xRatio;
  double h = glutBitmapHeight(Window::FONT) * yRatio;
  double x = -1.0 * w / 2.0;
  double y = -1.0 * h / 2.0;
  glColor4ub(Window::FG_COLOR[0], Window::FG_COLOR[1], Window::FG_COLOR[2], ProgressWindow::GetAlpha());
  glRasterPos2d(x, y + 4 * yRatio);
  glutBitmapString(Window::FONT, (const unsigned char*) ProgressWindow::message.c_str());
  // Draw border
  glLoadIdentity();
  glLineWidth(2.0f);
  glColor3ub(Window::BD_COLOR[0], Window::BD_COLOR[1], Window::BD_COLOR[2]);
  glBegin(GL_LINE_LOOP);
    glVertex2d(-1.0, -1.0); // bottom left
    glVertex2d(1.0, -1.0); // bottom right
    glVertex2d(1.0, 1.0); // top right
    glVertex2d(-1.0, 1.0); // top left
  glEnd();
  // Disable crop
  glDisable(GL_SCISSOR_TEST);
  }

// Handle window resize
void ProgressWindow::Reshape() 
  {
  ProgressWindow::width = RootWindow::width - Window::STATE_W - 3 * Window::PADDING;
  }

// Check simulator status
void ProgressWindow::Idle()
  {
  char str[256];
  switch (Demoddix::simulatorStatus())
    {
    // Simulator is running
    case Demoddix::RUN:
      Demoddix::Open();
      sprintf(str, "time = %lld", Demoddix::currentTime / 1000000);
      ProgressWindow::message = std::string(str);
      ProgressWindow::alpha -= 5;
      break;
    // Simulation was successful
    case Demoddix::OK:
      switch (Demoddix::displayStatus())
        {
        // Done tracing successfuly
        case Demoddix::OK:
          sprintf(str, "time = %lld", Demoddix::endTime / 1000000);
          ProgressWindow::message = std::string(str);
          ProgressWindow::alpha = 255;
          glutIdleFunc(NULL);
          break;
        // Done tracing with errors
        case Demoddix::KO:
          sprintf(str, "time = %lld | Tracing error!", Demoddix::endTime / 1000000);
          ProgressWindow::message = std::string(str);
          ProgressWindow::alpha = 255;
          glutIdleFunc(NULL);
          break;
        // Not done yet
        default:
          Demoddix::Open();
          sprintf(str, "time = %lld", Demoddix::currentTime / 1000000);
          ProgressWindow::message = std::string(str);
          ProgressWindow::alpha -= 5;
          break;
        }
      break;
    // If simulation failed, terminate its thread and set error status
    case Demoddix::KO:
      switch (Demoddix::displayStatus())
        {
        // Done tracing
        case Demoddix::OK:
        case Demoddix::KO:
          sprintf(str, "time = %lld | Simulation error!", Demoddix::currentTime / 1000000);
          ProgressWindow::message = std::string(str);
          ProgressWindow::alpha = 255;
          glutIdleFunc(NULL);
          break;
        // Not done yet
        default:
          Demoddix::Open();
          sprintf(str, "time = %lld | Simulation error!", Demoddix::currentTime / 1000000);
          ProgressWindow::message = std::string(str);
          ProgressWindow::alpha -= 5;
          break;
        }
      break;
    }
  // Update view
  glutPostRedisplay();
  }

// Update alpha transparency to create pulsing effect
unsigned char ProgressWindow::GetAlpha()
  {
  if (ProgressWindow::alpha < 127)
    {
    return (unsigned char) (255 - ProgressWindow::alpha);
    }
  else
    {
    return alpha;
    } 
  }
