/* Copyright 2013 Humboldt University of Berlin
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program. If not, see <http://www.gnu.org/licenses/>.
*
* Author: Mihal Brumbulli <mbrumbulli@gmail.com>
*/

#ifndef _DEMODDIX_H_
#define _DEMODDIX_H_

#include <cstdio>
#include <iostream>
#include <string>
#include <vector>
#include <deque>
#include <list>
#include <windows.h>

#include "GL/freeglut.h"

#include "tracer.h"
#include "window.h"
#include "root_window.h"
#include "state_window.h"
#include "state_color_window.h"
#include "message_window.h"
#include "message_color_window.h"
#include "toolbar_window.h"
#include "main_window.h"
#include "progress_window.h"

class State 
{
  public:
    std::string   name;     // State name
    unsigned int  color;    // Color index in Window
    int           priority; // State priority [0, 99]
    int           position; // Position in StateWindow
    
    // Constructors
    State() {}
    State(std::string name_, unsigned int color_, int priority_): name(name_), color(color_), priority(priority_), position(0) {}
};

class Process 
{
  public:
    std::string   name; // Process name
    
    // Constructors
    Process() {}
    Process(std::string name_): name(name_) {}
};

class Semaphore 
{
  public:
    std::string   name; // Semaphore name
    
    // Constructors
    Semaphore() {}
    Semaphore(std::string name_): name(name_) {}
};

class Message 
{
  public:
    std::string   name;	    // Message name
    unsigned int  color;    // Color index in Window
    int           position; // Position in MessageWindow
    bool          show;     // Toggle message display
    bool          active;   // Toggle display in legend (i.e., skip timers)
    
    // Constructors
    Message() {}
    Message(std::string name_, unsigned int color_): name(name_), color(color_), position(0), show(true), active(false) {}
};

class Node 
{
  public:
    std::string   name;	  // Node name (e.g., ip)
    double        x;      // Coordinate X
    double        y;      // Coordinate Y
    double        z;      // Coordinate Z
    unsigned int  state;  // State index
    
    // Constructors
    Node() {}
    Node(std::string name_, double x_, double y_, double z_, unsigned int state_): name(name_), x(x_), y(y_), z(z_), state(state_) {}
};

class Component 
{
  public:
    std::string   name; // Component name (e.g., port)
    unsigned int  node; // Node index
    
    // Constructors
    Component() {}
    Component(std::string name_, unsigned int node_): name(name_), node(node_) {}
};

class Event 
{
  public:
    long long     time; // Time of event
    unsigned int  nid;  // Node index
    unsigned int  cid;  // Component index
    fpos_t        pos;  // Trace file position
    bool          lost; // Lost packet
    
    // Constructors
    Event(long long time_, unsigned int nid_, unsigned int cid_, fpos_t pos_): time(time_), nid(nid_), cid(cid_), pos(pos_), lost(false) {}
};

class Packet 
{
  public:
    unsigned int    source;       // Sender node index
    unsigned int    destination;  // Receiver node index
    unsigned int    message;      // Message index
    unsigned int    id;           // Packet id
    Event         & event;        // Event 
    
    // Constructors
    Packet(unsigned int source_, unsigned int destination_, unsigned int message_, unsigned int id_, Event &event_): source(source_), destination(destination_), message(message_), id(id_), event(event_) {}
    // Is lost?
    bool lost() { return event.lost; }
};

class Demoddix 
{
  public:
    enum {INIT, RUN, OK, KO};       // Simulator and display status options
    
    static bool                   liveMode;         // Display events in read time
    
    static int                    simulatorStatus_; // Simulator status
    static int                    displayStatus_;   // Display status
    static CRITICAL_SECTION       statusLock_;      // Guard access to status variables
    static HANDLE                 simulatorThread;  // Simulator thread
    static std::string            simulatorExe;     // Simulator executable
    
    
    static unsigned int           bufferSize;       // Buffer size for reading
    static FILE                 * traceFile;        // Trace file
    static fpos_t                 tracePos;         // Current file position for live tracing
	
    static std::vector<State>     stateList;        // States list
    static std::vector<Message>   messageList;      // Messages list
    static std::vector<Process>   processList;      // Processes list
    static std::vector<Semaphore> semaphoreList;    // Semaphores list
    static std::vector<Node>      nodeList;         // Nodes list
    static std::vector<Component> componentList;    // Components list
    static std::list<Packet>      packetList;       // Packets list
	
    static std::deque<Event>      eventList;        // Events list
    static unsigned long          currentEvent;     // Current event index
	
    static long long              currentTime;      // Current time (ns)
    static long long              beginTime;        // Begin time (ns)
    static long long              endTime;          // End time (ns)
    static long long              stepTime;         // Step forward (or rewind) (ns) 
    
    static int simulatorStatus();
    static void simulatorStatus(int value);
    static int displayStatus();
    static void displayStatus(int value);
    
    static void Open();
    static void Close();
    static void Forward();
    static void Rewind();
    static void Next();
    static void Previous();
    static void Reset();
    static bool Front(Event &e, const char *buffer);
    static bool Back(Event &e, const char *buffer);
    static DWORD WINAPI RunSimulation(void *arg);
};

#endif
