/* Copyright 2013 Humboldt University of Berlin
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program. If not, see <http://www.gnu.org/licenses/>.
*
* Author: Mihal Brumbulli <mbrumbulli@gmail.com>
*/

#include <climits>
#include <string>

#include "demoddix.h"


int           MessageWindow::width;
int           MessageWindow::height;
int           MessageWindow::xPos;
int           MessageWindow::yPos;

int           MessageWindow::yMove            = 0;
unsigned long MessageWindow::selectedMessage  = ULONG_MAX;

// Create window
void MessageWindow::Create() 
  {
  MessageWindow::width = Window::STATE_W;
  MessageWindow::height =(RootWindow::height - (Window::PROGRESS_H + 4 * Window::PADDING)) / 2;
  MessageWindow::xPos = Window::PADDING;
  MessageWindow::yPos = Window::PROGRESS_H + 2 * Window::PADDING;
  // Crete color chooser
  MessageColorWindow::Create();
  }

// Draw messages
void MessageWindow::Display() 
  {
  // Display color chooser
  if (MessageWindow::selectedMessage != ULONG_MAX)
    {
    MessageColorWindow::Display();
    }
  // Vieport and crop
  glViewport(MessageWindow::xPos, MessageWindow::yPos, MessageWindow::width, MessageWindow::height);
  glScissor(MessageWindow::xPos, MessageWindow::yPos, MessageWindow::width, MessageWindow::height);
  glEnable(GL_SCISSOR_TEST);
  glLoadIdentity();
  glLineWidth(1.0f);
  // Draw
  if (Demoddix::displayStatus() != Demoddix::INIT)
    {
    // Ratio for mapping pixels to coordinates
    double xRatio = 2.0 / MessageWindow::width;
    double yRatio = 2.0 / MessageWindow::height;
    // Scroll
    glTranslated(0.0, MessageWindow::yMove * yRatio, 0.0);
    // Draw messages
    int y = Window::PADDING;
    glPointSize(Window::POINT);
    for (unsigned long i = 0; i < Demoddix::messageList.size(); ++i)
      {
      Message &m = Demoddix::messageList[i];
      if (!m.name.empty() && m.active) 
        {
        // Color
        glBegin(GL_POINTS);
          glColor3ub(Window::COLOR[m.color][0], Window::COLOR[m.color][1], Window::COLOR[m.color][2]);
          glVertex2d(-1.0 + (Window::PADDING + Window::POINT / 2.0) * xRatio, 1.0 - (y + Window::POINT / 2.0) * yRatio);
        glEnd();
        // Name and display toggle
        glColor3ub(Window::FG_COLOR[0], Window::FG_COLOR[1], Window::FG_COLOR[2]);
        glRasterPos2d(-1.0 + (2 * Window::PADDING + Window::POINT) * xRatio, 1.0 - (y + 4 + Window::POINT / 2.0) * yRatio);
        char str[256];
        if (m.show)
          { 
          sprintf(str, " [X] %s", m.name.c_str());
          }
        else
          {
          sprintf(str, " [ ] %s", m.name.c_str());
          }
        glutBitmapString(Window::FONT, (const unsigned char*) str);
        m.position = y;
        y += Window::POINT + Window::PADDING;
        }
      }
    }
  // Draw border
  glLoadIdentity();
  glLineWidth(2.0f);
  glColor3ub(Window::BD_COLOR[0], Window::BD_COLOR[1], Window::BD_COLOR[2]);
  glBegin(GL_LINE_LOOP);
    glVertex2d(-1.0, -1.0); // bottom left
    glVertex2d(1.0, -1.0); // bottom right
    glVertex2d(1.0, 1.0); // top right
    glVertex2d(-1.0, 1.0); // top left
  glEnd();
  // Disable crop
  glDisable(GL_SCISSOR_TEST);
  }

// Handle window resize
void MessageWindow::Reshape() 
  {
  MessageWindow::yMove = 0;
  MessageWindow::height =(RootWindow::height - (Window::PROGRESS_H + 4 * Window::PADDING)) / 2;
  // Update color chooser
  MessageColorWindow::Reshape();
  }

// Handle mouse click
void MessageWindow::OnMouseClick(int button, int state, int x, int y) 
  {
  // If there is a selected message, do nothing
  if (MessageWindow::selectedMessage != ULONG_MAX) 
    {
    if (x >= MessageColorWindow::xPos && x <= MessageColorWindow::xPos + MessageColorWindow::width && y >= MessageWindow::height + 3 * Window::PADDING && y <= MessageWindow::height + 3 * Window::PADDING + MessageColorWindow::height)
      {
      MessageColorWindow::OnMouseClick(button, state, x, y);
      }
    return;
    }
  // Update click coordinate based on window position (viewport)
  y -= MessageWindow::height + 2 * Window::PADDING;
  // If there is no selected message and ctrl + mouse left, toggle display
  // If mouse left, select that message and show color chooser
  if (button == GLUT_LEFT_BUTTON && state == GLUT_DOWN) 
    {
    y += MessageWindow::yMove;
    for (unsigned long i = 0; i < Demoddix::messageList.size(); ++i) 
      {
      Message& m = Demoddix::messageList[i];
      if (!m.name.empty()) 
        {
        if (y >= m.position && y <= m.position + Window::POINT) 
          {
          if (glutGetModifiers() == GLUT_ACTIVE_CTRL) 
            {
            m.show = !m.show;
            }
          else 
            {
            MessageWindow::selectedMessage = i;
            }
          break;
          }
        }
      }
    }
	// Scroll up
  else if (button == 3 && state == GLUT_DOWN) 
    {
    MessageWindow::yMove -= 4 * (Window::PADDING + Window::POINT);
    if (MessageWindow::yMove < 0) 
      {
      MessageWindow::yMove = 0;
      }
    }
  // Scroll down
  else if (button == 4 && state == GLUT_DOWN) 
    {
    int space = Window::POINT + Window::PADDING;
    int required = Demoddix::messageList.size() * space + Window::PADDING;
    int available = StateWindow::height;
    if (required > available) 
      {
      MessageWindow::yMove += 4 * space;
      if (MessageWindow::yMove > required - available) 
        {
        MessageWindow::yMove = required - available;
        }
      }
    }
  // Update view
  glutPostRedisplay();
  }
