/* Copyright 2013 Humboldt University of Berlin
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program. If not, see <http://www.gnu.org/licenses/>.
*
* Author: Mihal Brumbulli <mbrumbulli@gmail.com>
*/

#include <climits>
#include <string>
#include <algorithm>

#include "demoddix.h"


int           StateWindow::width;
int           StateWindow::height;
int           StateWindow::xPos;
int           StateWindow::yPos;

int           StateWindow::yMove          = 0;
unsigned long StateWindow::selectedState  = ULONG_MAX;

// Create window
void StateWindow::Create() 
  {
  StateWindow::width = Window::STATE_W;
  StateWindow::height = (RootWindow::height - (Window::PROGRESS_H + 4 * Window::PADDING)) / 2;
  StateWindow::xPos = Window::PADDING;
  StateWindow::yPos = StateWindow::height + Window::PROGRESS_H + 3 * Window::PADDING;
  // Create color chooser
  StateColorWindow::Create();
  }

// Draw states
void StateWindow::Display() 
  {
  // Display color chooser
  if (StateWindow::selectedState != ULONG_MAX)
    {
    StateColorWindow::Display();
    }
  // Viewport and crop
  glViewport(StateWindow::xPos, StateWindow::yPos, StateWindow::width, StateWindow::height);
  glScissor(StateWindow::xPos, StateWindow::yPos, StateWindow::width, StateWindow::height);
  glEnable(GL_SCISSOR_TEST);
  glLoadIdentity();
  glLineWidth(1.0f);
  // Draw
  if (Demoddix::displayStatus() != Demoddix::INIT)
    {
	  // Ratio for mapping pixels to coordinates
    double xRatio = 2.0 / StateWindow::width;
    double yRatio = 2.0 / StateWindow::height;
	  // Scroll
	  glTranslated(0.0, StateWindow::yMove * yRatio, 0.0);
	  // Draw states
    int y = Window::PADDING;
    glPointSize(Window::POINT);
    for (unsigned long i = 0; i < Demoddix::stateList.size(); ++i)
      {
      State &s = Demoddix::stateList[i];
      if (!s.name.empty()) 
        {
        // Color
        glBegin(GL_POINTS);
          glColor3ub(Window::COLOR[s.color][0], Window::COLOR[s.color][1], Window::COLOR[s.color][2]);
          glVertex2d(-1.0 + (Window::PADDING + Window::POINT / 2.0) * xRatio, 1.0 - (y + Window::POINT / 2.0) * yRatio);
        glEnd();
        // Name and priority
        glColor3ub(Window::FG_COLOR[0], Window::FG_COLOR[1], Window::FG_COLOR[2]);
        glRasterPos2d(-1.0 + (2 * Window::PADDING + Window::POINT) * xRatio, 1.0 - (y + 4 + Window::POINT / 2.0) * yRatio);
        char str[256];
        sprintf(str, " (%2d) %s", s.priority, s.name.c_str());
        glutBitmapString(Window::FONT, (const unsigned char*) str);
        s.position = y;
        y += Window::POINT + Window::PADDING;
        }
      }
    }
  // Draw border
  glLoadIdentity();
  glLineWidth(2.0f);
  glColor3ub(Window::BD_COLOR[0], Window::BD_COLOR[1], Window::BD_COLOR[2]);
  glBegin(GL_LINE_LOOP);
    glVertex2d(-1.0, -1.0); // bottom left
    glVertex2d(1.0, -1.0); // bottom right
    glVertex2d(1.0, 1.0); // top right
    glVertex2d(-1.0, 1.0); // top left
  glEnd();
  // Disable crop
  glDisable(GL_SCISSOR_TEST);
  }

// Handle window resize
void StateWindow::Reshape() 
  {
  StateWindow::yMove = 0;
  StateWindow::height = (RootWindow::height - (Window::PROGRESS_H + 4 * Window::PADDING)) / 2;
  StateWindow::yPos = StateWindow::height + Window::PROGRESS_H + 3 * Window::PADDING;
  // Update color chooser
  StateColorWindow::Reshape();
  }

// Handle mouse click
void StateWindow::OnMouseClick(int button, int state, int x, int y) 
  {
  // If there is a selected state, do nothing
  if (StateWindow::selectedState != ULONG_MAX) 
    {
    if (x >= StateColorWindow::xPos && x <= StateColorWindow::xPos + StateColorWindow::width && y >= 2 * Window::PADDING && y <= 2 * Window::PADDING + StateColorWindow::height)
      {
      StateColorWindow::OnMouseClick(button, state, x, y);
      }
    return;
    }
  // Update click coordinate based on window position (viewport)
  y -= Window::PADDING;
  // If there is no selected state and ctrl + mouse left, increment priority
  // If mouse left, select that state and show color chooser
  if (button == GLUT_LEFT_BUTTON && state == GLUT_DOWN) 
    {
    y += StateWindow::yMove;
    for (unsigned long i = 0; i < Demoddix::stateList.size(); ++i) 
      {
      State &s = Demoddix::stateList[i];
      if (!s.name.empty()) 
        {
        if (y >= s.position && y <= s.position + Window::POINT) 
          {
          if (glutGetModifiers() == GLUT_ACTIVE_CTRL) 
            {
            if (s.priority < 99) 
              {
              ++s.priority;
              }
            }
          else 
            {
            StateWindow::selectedState = i;
            }
          break;
          }
        }
      }
    }
  // If ctrl + mouse right, decrement priority
  else if (button == GLUT_RIGHT_BUTTON && state == GLUT_DOWN) 
    {
    if (glutGetModifiers() == GLUT_ACTIVE_CTRL) 
      {
      y += StateWindow::yMove;
      for (unsigned long i = 0; i < Demoddix::stateList.size(); ++i)
        {
        State &s = Demoddix::stateList[i];
        if (!s.name.empty()) 
          {
          if (y >= s.position && y <= s.position + Window::POINT) 
            {
            if (s.priority > 0) 
              {
              --s.priority;
              }
            break;
            }
          }
        }
      }
    }
  // Scroll up
  else if (button == 3 && state == GLUT_DOWN)
    {
    StateWindow::yMove -= 4 * (Window::PADDING + Window::POINT);
    if (StateWindow::yMove < 0) 
      {
      StateWindow::yMove = 0;
      }
    }
  // Scroll down
  else if (button == 4 && state == GLUT_DOWN) 
    {
    int space = Window::POINT + Window::PADDING;
    int required = Demoddix::stateList.size() * space + Window::PADDING;
    int available = StateWindow::height;
    if (required > available) 
      {
      StateWindow::yMove += 4 * space;
      if (StateWindow::yMove > required - available) 
        {
        StateWindow::yMove = required - available;
        }
      }
    }
  // Update view
  glutPostRedisplay();
  }
