/* PragmaDev RTDS Embos integration */
#include "RTDS_MACRO.h"
#include "RTDS_mem.h"

/* **************************************************************** *
 * RTDS_Malloc
 * **************************************************************** *
 * Allocate memory from the memory pool
 * **************************************************************** *
 * Parameters:
 *        size: the size of the memory to allocate
 * Returns:
 *        pointer to allocated memory if success 
 * **************************************************************** */
void * RTDS_Malloc( unsigned size )
    {
    void * pointer = NULL;
    
    /* Then we allocate memory */
    pointer = OS_malloc( size );
    
    /* If the pointer is null. Do not have enough memory */
    if( pointer == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_DONT_HAVE_ENOUGH_MEMORY );
        }
    
    return pointer;
    }
 
/* **************************************************************** *
 * RTDS_Free
 * **************************************************************** *
 * Release a previously allocated memory from the memory pool  
 * **************************************************************** *
 * Parameters: 
 *        ptr: the pointer to deallocate
 * Returns:
 *        Nothing
 * **************************************************************** */
void RTDS_Free( void * ptr )
    {
    OS_free( ptr );
    }

/* **************************************************************** *
 * RTDS_GetProcessInstanceId
 * **************************************************************** *
 * Returns the instance Id of process based on its number
 * **************************************************************** *
 * Parameters:
 *        processNumber: Number of the process defined in RTDS_gen.h
 * Returns:
 *        the process queue id if found
 * Error:
 *        System error call if not found and NULL returned
 * **************************************************************** */
RTDS_SdlInstanceId * RTDS_GetProcessInstanceId( int processNumber )
    {
    RTDS_GlobalProcessInfo * tmpGlobalProcessInfo = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    
    tmpGlobalProcessInfo = RTDS_globalProcessInfo;
    
    while ( tmpGlobalProcessInfo != NULL )
        {
        if ( tmpGlobalProcessInfo->sdlProcessNumber == processNumber )
            {
            break;
            }
        tmpGlobalProcessInfo = tmpGlobalProcessInfo->next;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    if ( tmpGlobalProcessInfo != NULL )
        {
        return ( tmpGlobalProcessInfo->mySdlInstanceId );
        }
    else
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_QUEUE_ID );
        return ( NULL );
        }
    }

/* **************************************************************** *
 * RTDS_MsgReceive
 * **************************************************************** *
 * Receives a message from a process instance
 * **************************************************************** *
 * Parameters:
 *        instance id to read the messsage from
 *        pointer on the message to receive
 *        size of message
 *        mode used to read (blocking or non blocking)
 * Returns:
 *       Nothing
 * **************************************************************** */
void RTDS_MsgReceive( RTDS_SdlInstanceId * instanceId, RTDS_MessageHeader ** message )
    {
    /* Receive the message */
    if( RTDS_MsgQueueReceive( instanceId->queueId, ( void ** )message, RTDS_QUEUE_MAX_MSG_LENGTH ) != RTDS_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_RECEIVE );  
        }
    }

/* **************************************************************** *
 * RTDS_MsgReceive
 * **************************************************************** *
 * Receives a message from a process's queue
 * **************************************************************** *
 * Parameters:
 *        queue id to read the messsage from
 *        pointer on the message to receive
 *        size of message
 *        mode used to read (blocking or non blocking)
 * Returns:
 *       RTDS_OK if everything is OK RTDS_ERROR otherwise
 * **************************************************************** */
int RTDS_MsgQueueReceive( RTDS_RtosQueueId queueId, void ** message, int size )
    {
    /* Receive the message */
    /* If the return size is not the same than the size give in parameter, return an error code */
    if( OS_Q_GetPtr( queueId, message ) != size )
        {
        return RTDS_ERROR;
        }
    /* Atfer receive a message we have to delete it from the queue */
    OS_Q_Purge( queueId );
    /* If not return an OK code */
    return RTDS_OK;
    }

/* **************************************************************** *
 * RTDS_MsgQueueSend
 * **************************************************************** *
 * Send a message in a process's queue
 * **************************************************************** *
 * Parameters:
 *        messageNumber representing a message name
 *        dataLength length of data pointed by pData
 *        pData  pointer on data sent with the message
 *        receiver message receiver queue address
 *        sender message sender queue address
 *        RTDS_currentContext currentContext of the task
 * Returns:
 *       Nothing
 * **************************************************************** */
void RTDS_MsgQueueSend
( long messageNumber,
 long dataLength,
 unsigned char * pData,
 RTDS_SdlInstanceId * receiver ,
 RTDS_SdlInstanceId * sender ,
 RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_MessageHeader RTDS_messageToSend;
    
    /* Fill the message fields */
    RTDS_messageToSend.messageNumber = messageNumber;
    RTDS_messageToSend.timerUniqueId = 0;
    RTDS_messageToSend.sender = sender;
    RTDS_messageToSend.receiver = receiver;
    RTDS_messageToSend.dataLength = dataLength;
    RTDS_messageToSend.pData = pData;
    RTDS_messageToSend.next = NULL;   
#ifdef RTDS_SIMULATOR
    RTDS_messageToSend.messageUniqueId = RTDS_GetMessageUniqueId();
#endif

    RTDS_SIMULATOR_TRACE( RTDS_messageSent, &RTDS_messageToSend, receiver, RTDS_currentContext );

    /* Send the message */
    if ( OS_Q_Put( receiver->queueId, ( void * )&RTDS_messageToSend, RTDS_QUEUE_MAX_MSG_LENGTH ) != 0 )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_SEND );
        }
    }

/* **************************************************************** *
 * RTDS_GetTimerUniqueId
 * **************************************************************** *
 * Get an available timer unique id.
 * Is used to set timers
 * **************************************************************** *
 * Parameters:
 *       timerList: points to the first element of the chained list 
 *       of RTDS_TimerState
 * Returns:
 *       The unique timer Id
 * **************************************************************** *
 * Context:
 *      The chained list of RTDS_TimerState is sorted by increasing
 *      unique timer id
 *      timerUniqueId = NULL means the message is not a timer; so it
 *      is avoided.
 * **************************************************************** */
long RTDS_GetTimerUniqueId( RTDS_TimerState * timerList )
    {
    RTDS_TimerState * RTDS_prevTimer = NULL;
    RTDS_TimerState * RTDS_timer = NULL;
    long newTimerId;

    RTDS_CRITICAL_SECTION_START;
    /* If list is empty take 1 */
    if ( timerList == NULL )
        {
        newTimerId = 1;
        }
    /* If 1 is available, take it */
    else if ( timerList->timerUniqueId != 1 )
        {
        newTimerId = 1;
        }
    else
        {
        /* If there is a gap in the timerId chained list, take an Id in the gap */
        RTDS_prevTimer = timerList;
        newTimerId = 0;
        for ( RTDS_timer = timerList->next; RTDS_timer != NULL; RTDS_timer = RTDS_timer->next )
            {
            if ( RTDS_timer->timerUniqueId != RTDS_prevTimer->timerUniqueId + 1 ) 
                {
                newTimerId = RTDS_prevTimer->timerUniqueId + 1;
                }
            RTDS_prevTimer = RTDS_timer;
            }
        /* No gap, let's take the next value */
        if ( newTimerId == 0 )
            {
            newTimerId = RTDS_prevTimer->timerUniqueId + 1;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* Check the counter did not go back to 0 */
    if ( newTimerId == 0 )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_NO_MORE_TIMER_UNIQUE_ID );
        }
    
    return newTimerId;
    }

/* **************************************************************** *
 * RTDS_StartTimer
 * **************************************************************** *
 * Starts a watchdog with the necessary parameters to create the
 * timer when it goes off.
 * **************************************************************** *
 * Parameters:
 *        instanceId: instance Id of the receiver
 *        timerNumber: Number of the timer
 *        timerUniqueId: Unique Id of the timer
 *        delay: delay of the timer
 *        timerStateList: Address of the list of timer
 *        currentContext: active task context
 * Returns:
 *        Nothing
 * **************************************************************** */
void RTDS_StartTimer
( RTDS_SdlInstanceId * instanceId,
 long timerNumber,
 long timerUniqueId,
 int delay,
 RTDS_TimerState ** pTimerStateList,
 RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_TimerState * timerState = NULL;
    RTDS_TimerState * previousTimerState = NULL;
    RTDS_TimerState * newTimerState = NULL;

    /* Allocate memory for this structure */
    newTimerState = ( RTDS_TimerState * )RTDS_MALLOC( sizeof( RTDS_TimerState ) );
    
    /* Initialize the new element */
    newTimerState->state = RTDS_TIMER_OK;
    newTimerState->timerNumber = timerNumber;
    newTimerState->timeoutValue = delay + RTDS_GetSystemTime();
    newTimerState->timerUniqueId = timerUniqueId;
    newTimerState->next = NULL;
    newTimerState->receiverId = instanceId;
    
    /* Allocate memory for the message to be sent by the watchDog function */
    newTimerState->watchDogParam = ( RTDS_MessageHeader * )RTDS_MALLOC( RTDS_QUEUE_MAX_MSG_LENGTH );
    
    /* Fill in the message parameters */
#ifdef RTDS_SIMULATOR
    newTimerState->watchDogParam->messageUniqueId = 0;
#endif
    newTimerState->watchDogParam->messageNumber = timerNumber;
    newTimerState->watchDogParam->timerUniqueId = timerUniqueId;
    newTimerState->watchDogParam->sender = instanceId;
    newTimerState->watchDogParam->receiver = instanceId;
    newTimerState->watchDogParam->dataLength = 0;
    newTimerState->watchDogParam->pData = NULL;
    newTimerState->watchDogParam->next = NULL;
    
   /* Insert the new element */
    if ( *pTimerStateList == NULL ) /* The list is empty */
        {
        *pTimerStateList = newTimerState;
        }
    /* The list is not empty */
    else
        {
        previousTimerState = NULL;
        for ( timerState = *pTimerStateList; timerState != NULL; timerState = timerState->next )
            {
            if ( timerState->timerUniqueId > timerUniqueId )
                {
                if ( previousTimerState == NULL )
                    {
                    *pTimerStateList = newTimerState;
                    newTimerState->next = timerState;
                    }
                else
                    {
                    previousTimerState->next = newTimerState;
                    newTimerState->next = timerState;
                    }
                break;
                }
            previousTimerState = timerState;
            }
        /* Inserted at the end of the list */
        if ( timerState == NULL )
            {
            previousTimerState->next = newTimerState;
            }
        }
    /* Watchdog creation */
    OS_CREATETIMER( ( OS_TIMER * )newTimerState, RTDS_WatchDogFunction, delay );
    /* trace the creation of the timer */
    RTDS_SIMULATOR_TRACE( RTDS_timerStarted, newTimerState, delay, RTDS_currentContext );
    }

/* **************************************************************** *
 * RTDS_WatchDogFunction
 * **************************************************************** *
 * Function called by Embos when the timer goes off
 * Send a timer message using the parameters given to the watchdog
 * when the timer is started
 * **************************************************************** *
 * Parameters:
*       Nothing
 * Returns:
 *      Nothing
 * **************************************************************** */
void RTDS_WatchDogFunction( void )
    {
    RTDS_TimerState * parameter = NULL;
    RTDS_MessageHeader * message = NULL;
    RTDS_SdlInstanceId * receiver = NULL;
    
    /* Retrieve the parameter initialized in RTDS_StartTimer function */
    parameter = ( RTDS_TimerState * )OS_GetpCurrentTimer();
    message = ( RTDS_MessageHeader * )parameter->watchDogParam;
    receiver = ( RTDS_SdlInstanceId * )parameter->receiverId;
    
    /* Send the message */
    if ( OS_Q_Put( receiver->queueId, ( void * )message, RTDS_QUEUE_MAX_MSG_LENGTH ) != 0 )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_Q_SEND_IN_WATCHDOG );
        }
    }

/* **************************************************************** *
 * RTDS_StopTimer
 * **************************************************************** *
 * Stops a timer in trying to delete the watchdog. If unsuccessfull
 * set it cancelled in the timer chained list verified by the RTDS
 * kernel
 * **************************************************************** *
 * Parameters:
 *        timerNumber: number of the timer
 *        pTimerStateList: pointing to the timer chained list
 *        RTDS_currentContext: currentContext of the task
 * Returns:
 *        Nothing
 * **************************************************************** */
void RTDS_StopTimer
( long timerNumber,
 RTDS_TimerState ** pTimerStateList,
 RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_TimerState * RTDS_timer = NULL;
    RTDS_TimerState * RTDS_prevTimer = NULL;
    RTDS_MessageHeader * RTDS_message = NULL;
    RTDS_MessageHeader * RTDS_prevMessage = NULL;

    RTDS_timer = *pTimerStateList;
    
    while( RTDS_timer != NULL )
        {
        if( ( RTDS_timer->timerNumber == timerNumber ) && ( RTDS_timer->state != RTDS_TIMER_CANCELLED ) )
            {
            RTDS_SIMULATOR_TRACE( RTDS_timerCancelled, RTDS_timer, NULL, RTDS_currentContext );
            
            /* Deallocate the parameter of this timer */
            RTDS_FREE( RTDS_timer->watchDogParam );
            RTDS_timer->watchDogParam = NULL;
            
            /* Stop and Delete the timer */
            OS_DeleteTimer( &RTDS_timer->watchDogId );
            
            /* If the we could not cancel the timer. Probably went off allready */
            /* Set it cancelled in the list */
            RTDS_timer->state = RTDS_TIMER_CANCELLED;
            
            /* Remove it from the list */
            /* Is the first of the list */
            if ( RTDS_prevTimer == NULL )
                {
                *pTimerStateList = RTDS_timer->next;
                }
            /* Not the first of the list */
            else
                {
                RTDS_prevTimer->next = RTDS_timer->next;
                }
            
            RTDS_FREE( RTDS_timer );
            RTDS_timer = NULL;
            }
        
        RTDS_prevTimer = RTDS_timer;
        RTDS_timer = RTDS_timer->next;
        }
    
    /* If execution gets here: the timer might be in the save queue */
    RTDS_prevMessage = NULL;
    for ( RTDS_message = RTDS_currentContext->readSaveQueue ; RTDS_message != NULL ; RTDS_message = RTDS_message->next )
        {
        if ( RTDS_message->messageNumber == timerNumber )
            {
            /* Remove it from the list */
            /* Is the first of the list */
            if ( RTDS_prevMessage == NULL )
                {
                RTDS_currentContext->readSaveQueue = RTDS_message->next;
                }
            /* Not the first of the list */
            else
                {
                RTDS_prevMessage->next = RTDS_message->next;
                }
            /* Deallocate memory of the message */
            RTDS_FREE( RTDS_message );
            RTDS_message = NULL;
            return;
            }
        RTDS_prevMessage = RTDS_message;
        }
    
    /* If execution gets here: the timer might be in the save queue */
    RTDS_prevMessage = NULL;
    for ( RTDS_message = RTDS_currentContext->writeSaveQueue ; RTDS_message != NULL ; RTDS_message = RTDS_message->next )
        {
        if ( RTDS_message->messageNumber == timerNumber )
            {
            /* Remove it from the list */
            /* Is the first of the list */
            if ( RTDS_prevMessage == NULL )
                {
                RTDS_currentContext->writeSaveQueue = RTDS_message->next;
                }
            /* Not the first of the list */
            else
                {
                RTDS_prevMessage->next = RTDS_message->next;
                }
            /* Deallocate memory of the message */
            RTDS_FREE( RTDS_message );
            RTDS_message = NULL;
            return;
            }
        RTDS_prevMessage = RTDS_message;
        }
    }

/* **************************************************************** *
 * RTDS_ProcessCreate
 * **************************************************************** *
 * Create a new SDL process and add a processInfo struct in the
 * process info chained list
 * **************************************************************** *
 * Parameters:
 *        name of the process
 *        process number
 *        address of the process function
 *        priority of the process
 *        RTDS_currentContext currentContext of the task
 * Returns:
 *       Nothing but updates the OFFSPRING of the caller
 * **************************************************************** */ 
void RTDS_ProcessCreate
( char * processName,
 int processNumber,
 void ( * functionAddress )( void ),
 unsigned char priority,
 RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    int numberOfInstance;
    RTDS_GlobalProcessInfo * processInfo = NULL;
    RTDS_GlobalProcessInfo * newProcessInfo = NULL;
    RTDS_GlobalProcessInfo * tmpProcessInfo = NULL;
    RTDS_GlobalProcessInfo * previousProcessInfo = NULL;
    RTDS_GlobalProcessInfo * tmpProcessInfoPtr = NULL;
    struct mappingOfTableOfStack * my_TmpMappingPtr;
    
    /* Allocate a new processInfo structure */
    newProcessInfo = ( RTDS_GlobalProcessInfo * )RTDS_MALLOC( sizeof( RTDS_GlobalProcessInfo ) );
    /* Allocate memory to the instance ID and queue control block of the task */
    newProcessInfo->mySdlInstanceId = ( RTDS_SdlInstanceId * )RTDS_MALLOC( sizeof( RTDS_SdlInstanceId ) );
    /* Allocate memory for the task control block */
    newProcessInfo->myRtosTaskId = ( RTDS_RtosTaskId )RTDS_MALLOC( sizeof( OS_TASK ) );
    
    /* Initialize the number of the instance */
    numberOfInstance = 0;
    
    RTDS_CRITICAL_SECTION_START;
    
    /* We point to the structure that hold the memory stack table */
    my_TmpMappingPtr = my_mappingOfTableOfStack;
    
    /* We need to look for a available stack */
    while( my_TmpMappingPtr != NULL )
        {
        /* If the stack is available */
        if( my_TmpMappingPtr->isFree )
            {
            /* We can return immediatly */
            break;
            }
        /* Then we loop again */
        numberOfInstance++;
        my_TmpMappingPtr = my_TmpMappingPtr->next;
        }
    
    /* If we find a available stack */
    if( my_TmpMappingPtr != NULL )
        {
        /* We set the field to unavailable */
        my_TmpMappingPtr->isFree = 0;
        }
    /* If not */
    else
        {
        my_TmpMappingPtr = ( struct mappingOfTableOfStack * )RTDS_MALLOC( sizeof( struct mappingOfTableOfStack ) );
        my_TmpMappingPtr->isFree = 0;
        my_TmpMappingPtr->numberOfTheInstance = numberOfInstance;
        my_TmpMappingPtr->next = my_mappingOfTableOfStack;
        my_mappingOfTableOfStack = my_TmpMappingPtr;
        }
    
    RTDS_CRITICAL_SECTION_STOP;
    
    /* Update the mapping between the SDL instance and the stack */
    /* Is useful when the RTDS_ProcessKill function need to change the value of the field isFree */
    newProcessInfo->my_MappingPtr = my_TmpMappingPtr;
    
    /* We check if the system try to create more than specified tasks */
    if( my_TmpMappingPtr->numberOfTheInstance >= MAX_INSTANCES )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_NO_MORE_STACK );
        }
    
    /* Allocate memory to the queue of the task */
    newProcessInfo->queueArea = RTDS_MALLOC( RTDS_QUEUE_MAX_MSG * RTDS_QUEUE_MAX_MSG_LENGTH );
    /* Allocate memory for the queue control block */
    newProcessInfo->mySdlInstanceId->queueId = ( RTDS_RtosQueueId )RTDS_MALLOC( sizeof( OS_Q ) );
    
    /* Create the communication queue */
    OS_Q_Create( newProcessInfo->mySdlInstanceId->queueId, newProcessInfo->queueArea, RTDS_QUEUE_MAX_MSG * RTDS_QUEUE_MAX_MSG_LENGTH );
    
    /* Fill fields which are relevant to the created task */
    newProcessInfo->parentSdlInstanceId = RTDS_currentContext->mySdlInstanceId;
    newProcessInfo->offspringSdlInstanceId = NULL;
    newProcessInfo->sdlProcessNumber = processNumber;
    newProcessInfo->sdlState = 0;
    newProcessInfo->next = NULL;
    newProcessInfo->currentMessage = NULL;
    newProcessInfo->timerList = NULL;
    newProcessInfo->readSaveQueue = NULL;
    newProcessInfo->writeSaveQueue = NULL;
    
#ifdef RTDS_SIMULATOR
    newProcessInfo->priority = priority;
#endif
    
    RTDS_CRITICAL_SECTION_START;
    
    /* Task creation without auto-start */
    OS_CREATETASK( newProcessInfo->myRtosTaskId, processName, functionAddress, priority, tableOfStack[ my_TmpMappingPtr->numberOfTheInstance ] );
    
    /* Udpdate the offspring field */
    RTDS_currentContext->offspringSdlInstanceId = newProcessInfo->mySdlInstanceId;
    /* Add the process information to the chained list pointed by the RTDS_globalProcessInfo global variable */
    if ( RTDS_globalProcessInfo != NULL )
        {
        processInfo = RTDS_globalProcessInfo;
        while( processInfo->next != NULL )
            {
            processInfo = processInfo->next;
            }
        processInfo->next = newProcessInfo; 
        }
    else
        {
        RTDS_globalProcessInfo = newProcessInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* Trace the process creation */
    RTDS_SIMULATOR_TRACE( RTDS_processCreated, newProcessInfo, NULL, RTDS_currentContext );
    }

/* **************************************************************** *
 * RTDS_ProcessForget
 *-------------------------------------------------------------------
 * Forgets a given process, freeing all associated information
 *
 * PARAMETERS:
 *  - RTDS_currentContext : The current context of the task to delete
 *  - deleteQueue: If true, the message queue for the process is destroyed
 * RETURNS:
 *  - The PID for the forgotten process or NULL if process was not found
 * **************************************************************** */
RTDS_RtosTaskId RTDS_ProcessForget( RTDS_GlobalProcessInfo * RTDS_currentContext, short deleteQueue )
    {
    RTDS_TimerState * RTDS_timer = NULL;
    RTDS_MessageHeader * message = NULL;
    RTDS_MessageHeader * tmpMessage = NULL;
    RTDS_GlobalProcessInfo * tmpProcessInfo = NULL;
    RTDS_GlobalProcessInfo * previousProcessInfo = NULL;
    RTDS_RtosTaskId processToKill = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    /* Let's free the current message if any */
    if ( RTDS_currentContext->currentMessage != NULL )
        {
        /* Free the message parameter structure if any */
        if (RTDS_currentContext->currentMessage->pData != NULL){
          RTDS_FREE(RTDS_currentContext->currentMessage->pData);
          }

        #ifdef RTDS_SIMULATOR
            /* Release the message unique id back to the pool */
            RTDS_ReleaseMessageUniqueId( RTDS_currentContext->currentMessage->messageUniqueId );
        #endif

        /* Set the current message to NULL */
        RTDS_currentContext->currentMessage = NULL;
        }
    
    /* Let's clean the timer chained list to free memory and stop watchdogs */
    for ( RTDS_timer = RTDS_currentContext->timerList; RTDS_timer != NULL; RTDS_timer = RTDS_currentContext->timerList )
        {
        /* Disable the timer */
        OS_StopTimer( &RTDS_timer->watchDogId );
        
        /* Deallocate the parameter of this timer */
        RTDS_FREE( RTDS_timer->watchDogParam );
        RTDS_timer->watchDogParam = NULL;
        
        /* Then delete it */
        OS_DeleteTimer( &RTDS_timer->watchDogId );
        
        /* Remove it from the list of timers */
        RTDS_currentContext->timerList = RTDS_timer->next;
        }
    
    /* Clean the save queue: free messages and message unique ids in the read and write save queues */
    for ( message = RTDS_currentContext->readSaveQueue; message != NULL; )
        {
    #ifdef RTDS_SIMULATOR
        RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
    #endif
        tmpMessage = message;
        message = message->next;
        RTDS_FREE( tmpMessage );
        tmpMessage = NULL;
        }
    
    for ( message = RTDS_currentContext->writeSaveQueue; message != NULL; )
        {
    #ifdef RTDS_SIMULATOR
        RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
    #endif
        tmpMessage = message;
        message = message->next;
        RTDS_FREE( tmpMessage );
        tmpMessage = NULL;
        }
    
    tmpProcessInfo = RTDS_globalProcessInfo;
    
    /* Then we can search in this list and thus release all memory it hold */ 
    while ( tmpProcessInfo != NULL )
        {
        if ( tmpProcessInfo->mySdlInstanceId == RTDS_currentContext->mySdlInstanceId )
            {
            /* Release the previously allocated memory for the queue area */
            RTDS_FREE( tmpProcessInfo->queueArea );
            /* Trace the process terminaison */
            RTDS_SIMULATOR_TRACE( RTDS_processDied, tmpProcessInfo, NULL, RTDS_currentContext );

            /* Update the process information chained list */
            if ( previousProcessInfo == NULL )
                {
                RTDS_globalProcessInfo = tmpProcessInfo->next;
                }
            else
                {
                previousProcessInfo->next = tmpProcessInfo->next;
                }
            processToKill = tmpProcessInfo->myRtosTaskId;
            tmpProcessInfo->my_MappingPtr->isFree = 1;
            if ( deleteQueue )
                {
                /* Discard all messages in the message queue */
                OS_Q_Clear( tmpProcessInfo->mySdlInstanceId->queueId );
                /* Release the previously allocated memory for the queue control block */
                RTDS_FREE( tmpProcessInfo->mySdlInstanceId->queueId );
                }
            RTDS_FREE( tmpProcessInfo->mySdlInstanceId );
            RTDS_FREE( tmpProcessInfo );
            break;
            }
        else
            {
            previousProcessInfo = tmpProcessInfo;
            tmpProcessInfo = tmpProcessInfo->next;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    return processToKill;
    }
/* **************************************************************** *
 * RTDS_ProcessKill
 * **************************************************************** *
 * Kills an SDL process and delete its queue and process info block
 * **************************************************************** *
 * Parameters:
 *        the address of the process current context
 * Returns:
 *        Nothing
 * **************************************************************** */
void RTDS_ProcessKill( RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_RtosTaskId pidToKill = NULL;
    
    /* Forget everything about process, delete its message queue and get its PID */
    pidToKill = RTDS_ProcessForget( RTDS_currentContext, 1 );
    
    /* Delete the task */
    if ( pidToKill != NULL )
        {
        /* Release the previously allocated memory for the task control block */
        RTDS_FREE( pidToKill );
        /* Before deleting the task we have to terminate it execution */
        OS_Terminate( NULL );
        }
    else
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_TASK_TO_DELETE_NOT_FOUND );
        }
    }

/* **************************************************************** *
 * RTDS_Semaphore_Create
 * **************************************************************** *
 * Create a semaphore
 * **************************************************************** *
 * Parameters:
 *        the name of the semaphore
 *        the option of the created semaphore 
 *        (specifies how tasks suspend on the semaphore by priority or fifo policy)
 *        the initial count of the semaphore (only for binary and counting semaphore)
 * Returns:
 *        the semaphore id of the created semaphore
 * **************************************************************** */
RTDS_SemaphoreId RTDS_Semaphore_Create( char * name, unsigned char initialCount, RTDS_SemaphoreType type )
    {
    RTDS_SemaphoreId sem = NULL;
    
    /* Allocate memory for the semaphore control block */
    sem = ( RTDS_SemaphoreId )RTDS_MALLOC( sizeof( RTDS_SemType ) );
    sem->typeName = type;
    
    switch( type )
        {
        case SEM_COUNTING :
        case SEM_BINARY :
            /* Create the semaphore */
            sem->semId.semaphore = ( OS_CSEMA * )RTDS_MALLOC( sizeof( OS_CSEMA ) );
            OS_CreateCSema( sem->semId.semaphore, initialCount );
            break;
        case SEM_MUTEX :
            /* Create the mutex */
            sem->semId.mutex = ( OS_RSEMA * )RTDS_MALLOC( sizeof( OS_RSEMA ) );
            OS_CREATERSEMA( sem->semId.mutex );
            break;
        default:
            RTDS_SYSTEM_ERROR( RTDS_ERROR_BAD_SEMAPHORE_TYPE );
            break;
        }
    
    /* Return the id of the semaphore */
    return sem;
    }

/* **************************************************************** *
 * RTDS_Sem_Delete
 * **************************************************************** *
 * Delete a Embos semaphore and delete its info from the semaphore
 * information chained list and free the related memory
 * **************************************************************** *
 * Parameters:
 *        the address of the semaphore to find the semaphore info block
 *        RTDS_currentContext currentContext of the task
 * Returns:
 *        Nothing
 * **************************************************************** */
void RTDS_Sem_Delete( RTDS_SemaphoreId semaphoreId, RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_GlobalSemaphoreInfo * semInfo = NULL;
    RTDS_GlobalSemaphoreInfo * semInfoPrev = NULL;
    
    RTDS_SIMULATOR_TRACE( RTDS_semaphoreDeleted, semaphoreId, 0, RTDS_currentContext );
    
    /* Remove the semaphore information from the chained list */
    semInfoPrev = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    for ( semInfo = RTDS_globalSemaphoreInfo ; semInfo != NULL ; semInfo = semInfo->next )
        {
        /* semaphore found */
        if ( semInfo->semaphoreId == semaphoreId )
            {
            if ( semInfoPrev == NULL )
                {
                RTDS_globalSemaphoreInfo = semInfo->next;
                }
            else
                {
                semInfoPrev->next = semInfo->next;
                }
            
            switch( semInfo->semaphoreId->typeName )
                {
                case SEM_COUNTING :
                case SEM_BINARY :
                    /* Delete the semaphore */
                    OS_DeleteCSema( semInfo->semaphoreId->semId.semaphore );
                    /* And release previously allocated memory */
                    RTDS_FREE( semInfo->semaphoreId->semId.semaphore );
                    break;
                case SEM_MUTEX :
                    /* Release previously allocated memory */
                    RTDS_FREE( semInfo->semaphoreId->semId.mutex );
                    break;
                default:
                    RTDS_SYSTEM_ERROR( RTDS_ERROR_BAD_SEMAPHORE_TYPE );
                    break;
                }
            
            /* Release previously allocated memory */
            RTDS_FREE( semInfo->semaphoreId );
            semInfo->semaphoreId = NULL;
            RTDS_FREE( semInfo );
            semInfo = NULL;
            break;
            }
        semInfoPrev = semInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    }

/* **************************************************************** *
 * RTDS_SemaphoreIdTake
 * **************************************************************** *
 * Attempt to take a semaphore from its id
 * **************************************************************** *
 * Parameters:
 *        the id of the semaphore
 *        timeout value
 *        RTDS_currentContext currentContext of the task
 * Returns:
 *        status
 * **************************************************************** */
RTDS_SemaphoreStatus RTDS_SemaphoreIdTake( RTDS_SemaphoreId semaphoreId, RTDS_SemaphoreTimeout TIME_OUT, RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    int retval = 0;
    RTDS_SemaphoreStatus tmpStatus = RTDS_ERROR;
    
    /* Trace for semaphore take attempt */
    RTDS_SIMULATOR_TRACE( RTDS_semTakeAttempt, semaphoreId, TIME_OUT, RTDS_currentContext );
    
    switch( semaphoreId->typeName )
        {
        case SEM_COUNTING :
        case SEM_BINARY :
            /* If we have to wait for the semaphore forever */
            if( TIME_OUT == RTDS_SEMAPHORE_TIME_OUT_FOREVER )
                {
                /* We wait for the semaphore until is available */
                OS_WaitCSema( semaphoreId->semId.semaphore );
                /* Then we set the the status to OK */
                retval = 1;
                }
            /* If the semaphore take attempt is temporized */
            else
                {
                /* Then we wait for the semaphore for the specified time */
                retval = OS_WaitCSemaTimed( semaphoreId->semId.semaphore, TIME_OUT );
                }
            break;
        case SEM_MUTEX :
            /* If we have to wait for the semaphore forever */
            if( TIME_OUT == RTDS_SEMAPHORE_TIME_OUT_FOREVER )
                {
                /* Then we set the the status to OK */
                retval = OS_Use( semaphoreId->semId.mutex );
                }
            /* If the semaphore take attempt is temporized */
            else
                {
                /* Raise an error if a timeout is specified because mutex doesnt support timeout */
                RTDS_SYSTEM_ERROR( RTDS_ERROR_MUTEX_NOT_SUPPORT_TIMEOUT );
                }
            break;
        default:
            RTDS_SYSTEM_ERROR( RTDS_ERROR_BAD_SEMAPHORE_TYPE );
            break;
        }
    
    /* Check the status for trace purpose */
    if ( retval != 0 )
        {
        tmpStatus = RTDS_OK;
        RTDS_SIMULATOR_TRACE( RTDS_semTakeSucceded, semaphoreId, -1, RTDS_currentContext );
        }
    else
        {
        RTDS_SIMULATOR_TRACE( RTDS_semTakeTimedOut, semaphoreId, -1, RTDS_currentContext );
        }
    /* Then return the status */
    return tmpStatus;
    }

/* **************************************************************** *
 * RTDS_SemaphoreIdGive
 * **************************************************************** *
 * Attempt to release a semaphore from its id
 * **************************************************************** *
 * Parameters:
 *        the id of the semaphore
 * Returns:
 *        Nothing
 * **************************************************************** */
void RTDS_SemaphoreIdGive( RTDS_SemaphoreId semaphoreId )
    {
    switch( semaphoreId->typeName )
        {
        case SEM_COUNTING :
        case SEM_BINARY :
            /* Release the semaphore */
            OS_SignalCSema( semaphoreId->semId.semaphore );
            break;
        case SEM_MUTEX :
            /* Release the mutex */
            OS_Unuse( semaphoreId->semId.mutex );
            break;
        default:
            RTDS_SYSTEM_ERROR( RTDS_ERROR_BAD_SEMAPHORE_TYPE );
            break;
        }
    }

/* **************************************************************** *
 * RTDS_Sem_Info_Insert
 * **************************************************************** *
 * Inserts a semaphoreInfo struct in the semaphore info chained list
 * **************************************************************** *
 * Parameters:
 *        semaphore number
 *        id of the semaphore
 *        RTDS_currentContext currentContext of the task
 * Returns:
 *        The semaphore id so that the user can use the semaphore id directly
 * **************************************************************** */
RTDS_SemaphoreId RTDS_Sem_Info_Insert( int semaphoreNumber, RTDS_SemaphoreId semaphoreId, RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_GlobalSemaphoreInfo * semInfo = NULL;
    RTDS_GlobalSemaphoreInfo * newSemInfo = NULL;
    
    /* Allocate and fill in a new semInfo structure */
    newSemInfo = ( RTDS_GlobalSemaphoreInfo * )RTDS_MALLOC( sizeof( RTDS_GlobalSemaphoreInfo ) );       
    newSemInfo->semaphoreNumber = semaphoreNumber;
    newSemInfo->next = NULL;
    newSemInfo->semaphoreId = semaphoreId;
    
    RTDS_CRITICAL_SECTION_START;
    /* Add the semaphore information to the chained list pointed by the RTDS_globalSemaphoreInfo global variable */
    if ( RTDS_globalSemaphoreInfo == NULL )
        {
        RTDS_globalSemaphoreInfo = newSemInfo;
        }
    else
        {
        /* Let's get to the end of the list */
        for ( semInfo = RTDS_globalSemaphoreInfo; semInfo->next != NULL; semInfo = semInfo->next );
        semInfo->next = newSemInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* Trace the semaphore creation */
    RTDS_SIMULATOR_TRACE( RTDS_semaphoreCreated, semaphoreId, -1, RTDS_currentContext );
    
    /* Return the semaphore id */
    return semaphoreId;
    }

/* **************************************************************** *
 * RTDS_GetSemaphoreId
 * **************************************************************** *
 * Gets the id of a semaphore from its number (name)
 * **************************************************************** *
 * Parameters:
 *        semaphore number representing its name
 * Returns:
 *        the id of the semaphore
 * **************************************************************** */
RTDS_SemaphoreId RTDS_GetSemaphoreId( int semaphoreNumber )
    {
    RTDS_GlobalSemaphoreInfo * semInfo = NULL;
    RTDS_SemaphoreId foundSemaphoreId = NULL;

    RTDS_CRITICAL_SECTION_START;
    for ( semInfo = RTDS_globalSemaphoreInfo; semInfo != NULL; semInfo = semInfo->next )
        {
        /* Semaphore found */
        if ( semInfo->semaphoreNumber == semaphoreNumber )
            {
            foundSemaphoreId = semInfo->semaphoreId;
            break;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* Return the semaphore id */
    return foundSemaphoreId;
    }

/* **************************************************************** *
 * RTDS_TransitionCleanUp
 * **************************************************************** *
 * Called at the end of transitions:
 * - frees message buffer if valid
 * - re-organize save queue if state has changed
 * **************************************************************** *
 * Parameters:
 *      Pointer to the current context of the task
 *      Previous sdl state
 * Returns:
 *      Nothing
 * **************************************************************** */
void RTDS_TransitionCleanUp( RTDS_GlobalProcessInfo * RTDS_currentContext, int RTDS_sdlStatePrev )
    {
    /* Free message buffer if valid */
    if ( RTDS_currentContext->currentMessage != NULL )
        {
        /* Free the message parameter structure if any */
        if (RTDS_currentContext->currentMessage->pData != NULL){
          RTDS_FREE(RTDS_currentContext->currentMessage->pData);
          }
        #ifdef RTDS_SIMULATOR
            /* Release the message unique id back to the pool */
            RTDS_ReleaseMessageUniqueId( RTDS_currentContext->currentMessage->messageUniqueId );
        #endif
        RTDS_currentContext->currentMessage = NULL;
        } /* End of if ( RTDS_currentContext->currentMessage != NULL ) */
    
    /* If SDL state has changed and messages have been saved: reorganise the save queue */
    if ( ( RTDS_currentContext->sdlState != RTDS_sdlStatePrev ) && ( RTDS_currentContext->writeSaveQueue != NULL ) )
        {
        RTDS_MessageHeader * message = NULL;
        /* Let's get to the end of the save queue */
        for ( message = RTDS_currentContext->writeSaveQueue; message->next != NULL; message = message->next );
        message->next = RTDS_currentContext->readSaveQueue;
        RTDS_currentContext->readSaveQueue = RTDS_currentContext->writeSaveQueue;
        RTDS_currentContext->writeSaveQueue = NULL;
        }
    }

/* **************************************************************** *
 * RTDS_GetSystemTime
 * **************************************************************** *
 * As its name states...
 * **************************************************************** *
 * Parameters:
 *      None
 * Returns:
 *      System tick count value
 * **************************************************************** */
int RTDS_GetSystemTime( void )
    {
    return OS_GetTime();
    }

/* **************************************************************** *
 * RTDS_GetActiveThread
 * **************************************************************** *
 * The simulator call this function and reads the returned value
 * thus he know the id of current task
 * **************************************************************** *
 * Parameters:
 *      None
 * Returns:
 *      The id of the current task
 * **************************************************************** */
RTDS_RtosTaskId RTDS_GetActiveThread( void )
    {
    return OS_GetpCurrentTask();
    }

/* **************************************************************** *
 * **************************************************************** *
 * **************************************************************** *
 * THE CODE BELOW IS ONLY USED TO DEBUG WITH RTDS SDL-RT DEBUGGER
 * **************************************************************** *
 * **************************************************************** *
 * **************************************************************** */

#ifdef RTDS_SIMULATOR

unsigned char * RTDS_globalMessageUniqueIdPool = NULL;
unsigned long RTDS_globalSystemTime;
RTDS_GlobalTraceInfo RTDS_globalTraceEntry = { RTDS_systemError, NULL, 0, NULL };
char * RTDS_globalPrintableParameters = NULL;

/* **************************************************************** *
 * RTDS_GetMessageUniqueId
 * **************************************************************** *
 * Gets a message unique id for the simulator
 * **************************************************************** *
 * Parameters:
 *      None
 * Returns:
 *      the message unique id (minimum is 1)
 * **************************************************************** */
unsigned long RTDS_GetMessageUniqueId( void )
    {
    unsigned char * index = NULL;
    long uniqueByteId;
    long uniqueBitId;
    
    index = RTDS_globalMessageUniqueIdPool;
    
    RTDS_CRITICAL_SECTION_START;
    for ( uniqueByteId = 0; uniqueByteId < RTDS_MESSAGE_UNIQUE_ID_POOL_SIZE; uniqueByteId++ )
        {
        if ( *index != 0xFF )
            {
            for ( uniqueBitId = 0; uniqueBitId < 8; uniqueBitId++ )
                {
                if ( ( ( 1 << uniqueBitId ) & *index ) == 0 )
                    {
                    *index = *index | ( 1 << uniqueBitId );
                    RTDS_CRITICAL_SECTION_STOP;
                    return ( 8 * uniqueByteId + uniqueBitId + 1 );
                    }
                }
            }
        index++;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* All bits are set... No more message unique id */
    RTDS_SYSTEM_ERROR( RTDS_ERROR_NO_MORE_MSG_UNIQUE_ID );
    
    return 0;
    }

/* **************************************************************** *
 * RTDS_ReleaseMessageUniqueId
 * **************************************************************** *
 * Make a message unique id available from the pool
 * **************************************************************** *
 * Parameters:
 *      message unique id
 * Returns:
 *      Nothing
 * **************************************************************** */
void RTDS_ReleaseMessageUniqueId( unsigned long messageUniqueId )
    {
    unsigned char * index = NULL;
    
    /* probably a timer */
    if ( messageUniqueId == 0 )
        {
        return;
        }
    messageUniqueId -= 1;
    index = RTDS_globalMessageUniqueIdPool;
    index += ( unsigned char )( messageUniqueId / 8 );
    
    RTDS_CRITICAL_SECTION_START;
    ( *index ) = ( *index ) & ~( 1 << messageUniqueId % 8 );
    RTDS_CRITICAL_SECTION_STOP;
    }

/* **************************************************************** *
 * RTDS_DummyTraceFunction
 * **************************************************************** *
 * As its name states... The simulator sets a breakpoint on this
 * function and reads the RTDS_globalTraceEntry variable to see
 * what happened
 * **************************************************************** *
 * Parameters:
 *      None
 * Returns:
 *      Nothing
 * **************************************************************** */
void RTDS_DummyTraceFunction( void )
  {
  }

/* * **************************************************************** *
 * RTDS_SimulatorMsgQueueSend
 * **************************************************************** *
 * Send a message in a process's queue from the simulator
 * Basically the same as RTDS_MsgQueueSend but it gets the
 * MsgUniqueId as a parameter and does not call the dummy trace
 * function
 * **************************************************************** *
 * Parameters:
 *        messageNumber representing a message name
 *        dataLength length of data pointed by pData
 *        pData pointer on data sent with the message
 *        receiver message receiver queue address
 *        sender message sender queue address
 *        MsgUniqueId unique id of the message
 * Returns:
 *        Nothing
 * *****************************************************************/
void RTDS_SimulatorMsgQueueSend( 
    long                 messageNumber,
    long                 dataLength,
    unsigned char      * pData,
    RTDS_SdlInstanceId * receiver,
    RTDS_SdlInstanceId * sender,
    long                 MsgUniqueId )
    {
    RTDS_MessageHeader RTDS_messageToSend;
    
    RTDS_messageToSend.messageNumber = messageNumber;
    RTDS_messageToSend.timerUniqueId = 0;
    RTDS_messageToSend.sender = sender;
    
    if ( dataLength != 0 )
        {
        RTDS_stringToMessageData( messageNumber, &( RTDS_messageToSend.dataLength ), ( void ** )&( RTDS_messageToSend.pData ), ( char * )pData );
        RTDS_FREE( pData );
        pData = NULL;
        }
    else
        {
        RTDS_messageToSend.dataLength = 0;
        RTDS_messageToSend.pData = NULL;
        }
    
    RTDS_messageToSend.next = NULL;
    RTDS_messageToSend.messageUniqueId = MsgUniqueId;
    
    /* Specific for the backTrace since the original trace comes from the SDL-RT debugger */
#ifdef RTDS_BACK_TRACE_MAX_EVENT_NUM
    RTDS_globalTraceEntry.event = RTDS_messageSent;
    RTDS_globalTraceEntry.eventParameter1 = ( void * )&RTDS_messageToSend;
    RTDS_globalTraceEntry.eventParameter2 = ( long )receiver;
    RTDS_globalTraceEntry.currentContext = ( RTDS_GlobalProcessInfo * )NULL;
    RTDS_TraceAdd();
#endif
    /* send the message to receiver */
    if( OS_Q_Put( receiver->queueId, ( void * )&RTDS_messageToSend, RTDS_QUEUE_MAX_MSG_LENGTH ) != RTDS_QUEUE_MAX_MSG_LENGTH )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_SEND );
        }
    }

#endif

