#define __STDC_FORMAT_MACROS
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <inttypes.h>

#include "RTDS_Proxy.h"
#include "RTDS_Scheduler.h"
#include "RTDS_Deployment.h"
#include "RTDS_BasicTypes.h"
#include "RTDS_InternalConstants.h"
#include "RTDS_OS.h"

RTDS_Proxy::RTDS_Proxy(RTDS_Scheduler *parentScheduler): RTDS_Proc(parentScheduler)
  {
  // Setup context
  RTDS_currentContext = (RTDS_GlobalProcessInfo *) malloc(sizeof(RTDS_GlobalProcessInfo));
  memset(RTDS_currentContext, 0, sizeof(RTDS_GlobalProcessInfo));
  // Reset sockets
  listeningSocket = 0;
  memset(&slots, 0, sizeof(slots));
  }

RTDS_Proxy::~RTDS_Proxy()
  {
  // Close sockets
  int i;
  for (i = 0; i < RTDS_SETSIZE; ++i)
    {
    if (slots[i] != 0)
      {
      slots[i]->Close();
      }
    }
  }

short RTDS_Proxy::RTDS_continuousSignals(int *lowestPriority) 
  {
  return 0;
  }

short RTDS_Proxy::RTDS_executeTransition(RTDS_MessageHeader *currentMessage) 
  {
  // Startup message
  if (currentMessage == NULL)
    {
    // If already listening
    if (listeningSocket != 0)
      {
      // Done
      return 0;
      }
    // Trace
    fprintf(RTDS_traceFile, "    <taskCreated nId=\"n%u\" cId=\"c%u\" time=\"%"PRId64"\" creatorId=\"%u\" pName=\"p%d\" creatorName=\"p%d\" pId=\"%u\" />\n",
      RTDS_parentScheduler->component->node->nodeUniqueId,
      RTDS_parentScheduler->component->componentUniqueId,
      RTDS_TickGet(), 
      RTDS_currentContext->mySdlInstanceId.instanceNumber, 
      RTDS_currentContext->mySdlInstanceId.processNumber, 
      RTDS_currentContext->mySdlInstanceId.processNumber, 
      RTDS_currentContext->mySdlInstanceId.instanceNumber
    );
    // Get address and port information local scheduler
    char *addr = RTDS_parentScheduler->component->node->nodeId;
    unsigned short port = (unsigned short) atoi(RTDS_parentScheduler->component->componentId);
    // Create listening socket
    listeningSocket = RTDS_create(addr, port, 1, NULL);
    // If error, do nothing
    if (listeningSocket == 0)
      {
      // Terminate
      return 1;
      }
    // Remember it
    RTDS_insert(listeningSocket);
    }
  // Normal message
  else
    {
    // Look for peer scheduler
    RTDS_DeplComponent *component = RTDS_parentScheduler->component->node->getComponent(currentMessage->receiver.componentNumber);
    // If component wasn't found for some reason (shouldn't happen)
    if (component == NULL)
      {
      // Discard message
      if (currentMessage->pData != NULL)
        {
        free(currentMessage->pData);
        }
      free(currentMessage);
      // Never terminate
      return 0;
      }
    // Trace packet
    fprintf(RTDS_traceFile, "    <information nId=\"n%u\" cId=\"c%u\" time=\"%"PRId64"\" pName=\"p%d\" pId=\"%u\" message=\"out to %u\" />\n",
      RTDS_parentScheduler->component->node->nodeUniqueId,
      RTDS_parentScheduler->component->componentUniqueId,
      RTDS_TickGet(), 
      RTDS_currentContext->mySdlInstanceId.processNumber, 
      RTDS_currentContext->mySdlInstanceId.instanceNumber, 
      component->node->nodeUniqueId
    );
    fprintf(RTDS_traceFile, "    <packetSent nId=\"n%u\" cId=\"c%u\" time=\"%"PRId64"\" nReceiver=\"n%u\" pktName=\"m%d\" pktId=\"%d\" />\n",
      RTDS_parentScheduler->component->node->nodeUniqueId,
      RTDS_parentScheduler->component->componentUniqueId,
      RTDS_TickGet(), 
      component->node->nodeUniqueId, 
      currentMessage->messageNumber,
      currentMessage->messageUniqueId
    );
    // Get address and port information
    char *addr = component->node->nodeId;
    unsigned short port = (unsigned short) atoi(component->componentId);
    // Create connection socket
    short connected = 1;
    ns3::Ptr<ns3::Socket> sock = RTDS_create(addr, port, 0, &connected);
    // Error creating socket
    if (sock == 0)
      {
      // Discard message
      if (currentMessage->pData != NULL)
        {
        free(currentMessage->pData);
        }
      free(currentMessage);
      }
    // Socket created
    else
      {
      // Keep it in timerUniqueId
      currentMessage->timerUniqueId = RTDS_insert(sock);
      // Socket registered
      if (currentMessage->timerUniqueId > 0)
        {
        // Save message (to be sent when connection succeeds)
        RTDS_msgSave(currentMessage);
        }
      // Socket not registered
      else
        {
        // RTDS_ERROR ???
        // Discard message
        if (currentMessage->pData != NULL)
          {
          free(currentMessage->pData);
          }
        free(currentMessage);
        // Close socket
        sock->Close();
        }
      }
    }
  // Never terminate
  return 0;
  }

ns3::Ptr<ns3::Socket> RTDS_Proxy::RTDS_create(char *address, unsigned short port, short listening, short *connected)
  {
  // Create socket
  ns3::Ptr<ns3::Socket> sock = ns3::Socket::CreateSocket(ns3::NodeList::GetNode(RTDS_parentScheduler->component->node->nodeUniqueId), ns3::TcpSocketFactory::GetTypeId());
  if (sock == 0)
    {
    // RTDS_ERROR ???
    return 0;
    }
  // Server socket
  if (listening)
    {
    // Bind
    if (sock->Bind(ns3::InetSocketAddress(address, port)) < 0)
      {
      // RTDS_ERROR ???
      sock->Close();
      return 0;
      }
    // Listen
    if (sock->Listen() < 0)
      {
      // RTDS_ERROR ???
      sock->Close();
      return 0;
      }
    // Setup callbacks
    sock->SetAcceptCallback(ns3::MakeNullCallback<bool, ns3::Ptr<ns3::Socket>, const ns3::Address &>(), ns3::MakeCallback(&RTDS_Proxy::RTDS_accept, this));
    }
  // Client socket
  else
    {
    if (sock->Connect(ns3::InetSocketAddress(address, port)) < 0)
      {
      // RTDS_ERROR ???
      sock->Close();
      return 0;
      }
    //Setup callbacks
    sock->SetConnectCallback(ns3::MakeCallback(&RTDS_Proxy::RTDS_connectOk, this), ns3::MakeCallback(&RTDS_Proxy::RTDS_connectKo, this));
    // Socket never connects immediately
    *connected = 0;
    }
  // Return socket
  return sock;
  }

void RTDS_Proxy::RTDS_accept(ns3::Ptr<ns3::Socket> sock, const ns3::Address &addr)
  {
  // Setup callback
  sock->SetRecvCallback(ns3::MakeCallback(&RTDS_Proxy::RTDS_receive, this));
  }

void RTDS_Proxy::RTDS_connectOk(ns3::Ptr<ns3::Socket> sock)
  {
  // Setup callback
  sock->SetSendCallback(ns3::MakeCallback(&RTDS_Proxy::RTDS_send, this));
  // Send pending message
	RTDS_send(sock, sock->GetTxAvailable());
  }

void RTDS_Proxy::RTDS_connectKo(ns3::Ptr<ns3::Socket> sock)
  {
  // Discard pending message
  RTDS_MessageHeader *currMsg = RTDS_pending(sock);
  if (currMsg != NULL)
    {
    free(currMsg->pData);
    free(currMsg);
    }
  // Forget socket
  RTDS_remove(sock);
  // Close it
  sock->Close();
  }

void RTDS_Proxy::RTDS_send(ns3::Ptr<ns3::Socket> sock, uint32_t bytes)
  {
  // Look for pending message to be sent
  RTDS_MessageHeader *currMsg = RTDS_pending(sock);
  // Message not found (shouldn't happen)
  if (currMsg == NULL)
    {
    // Forget socket
    RTDS_remove(sock);
    // Close it
    sock->Close();
    // Done
    return;
    }
  long dataLength;
  unsigned char *pData;
  // First attempt at sending
  if (currMsg->messageNumber > 0)
    {
    // Format and pack data
    dataLength = 40 + currMsg->dataLength;
    pData = (unsigned char *) malloc(dataLength);
    uint32_t packetHeader[10] = {
      (uint32_t) dataLength,
      (uint32_t) currMsg->messageNumber,
      (uint32_t) currMsg->messageUniqueId,
      (uint32_t) currMsg->sender.processNumber,
      (uint32_t) currMsg->sender.instanceNumber,
      (uint32_t) currMsg->sender.componentNumber,
      (uint32_t) currMsg->receiver.processNumber,
      (uint32_t) currMsg->receiver.instanceNumber,
      (uint32_t) currMsg->receiver.componentNumber,
      (uint32_t) currMsg->dataLength
    };
    memcpy(pData, packetHeader, 40);
    memcpy(pData + 40, currMsg->pData, currMsg->dataLength);
    // Update message with formatted data
    if (currMsg->pData != NULL)
      {
      free(currMsg->pData);
      }
    currMsg->messageNumber = -1;
    currMsg->dataLength = dataLength;
    currMsg->pData = pData;
    } 
  // Max bytes to send
  bytes = (uint32_t) currMsg->dataLength < bytes ? (uint32_t) currMsg->dataLength : bytes;
  // Send failed
  if (sock->Send(currMsg->pData, bytes, 0) != (int) bytes)
    {
    // RTDS_ERROR ???
    // Discard message
    free(currMsg->pData);
    free(currMsg);
    // Forget socket
    RTDS_remove(sock);
    // Close it
    sock->Close();
    }
  // Send successful
  else
    {
    currMsg->dataLength -= bytes;
    // There is still data to be sent
    if (currMsg->dataLength > 0)
      {
      // Update message to contain only the data to be sent
      pData = (unsigned char *) malloc(currMsg->dataLength);
      memcpy(pData, currMsg->pData + bytes, currMsg->dataLength);
      free(currMsg->pData);
      currMsg->pData = pData;
      // Save message so that can be handled later
      RTDS_msgSave(currMsg); 
      }
    // Message was sent entirely
    else
      {
      // Discard message
      free(currMsg->pData);
      free(currMsg);
      // Forget socket
      RTDS_remove(sock);
      }
    }
  }

void RTDS_Proxy::RTDS_receive(ns3::Ptr<ns3::Socket> sock)
  {
  // Look for pending message to be received
  RTDS_MessageHeader *currMsg = RTDS_pending(sock);
  // If first attempt at receiving, create message
  if (currMsg == NULL)
    {
    // Get total size
    uint32_t dataLength;
    int bytes = sock->Recv((uint8_t *) &dataLength, 4, 0);
    // Success
    if (bytes > 0)
      {
      // Register socket
      int index = RTDS_insert(sock);
      // Limit reached
      if (index == 0)
        {
        // Close socket
        sock->Close();
        // Done
        return;
        }
      // Create message
      currMsg = (RTDS_MessageHeader *) malloc(sizeof(RTDS_MessageHeader));
      currMsg->timerUniqueId = index;
      // Actual size is stored in messageNumber
      currMsg->messageNumber = 4;
      // Total size is stored in dataLength
      currMsg->dataLength = (long) dataLength;
      // Allocate buffer
      currMsg->pData = (unsigned char *) malloc(currMsg->dataLength);
      // First is length (not used but needed as part of the header)
      memcpy(currMsg->pData, &dataLength, 4);
      }
    // Error ??? (no idea why it returns 0)
    else
      {
      // Close socket
      sock->Close();
      // Done
      return;
      }
    }
  // Receive
  int bytes = sock->Recv(currMsg->pData + currMsg->messageNumber, currMsg->dataLength - currMsg->messageNumber, 0);
  // Error
  if (bytes <= 0)
    {
    // RTDS_ERROR ???
    // Discard message
    free(currMsg->pData);
    free(currMsg);
    // Forget socket
    RTDS_remove(sock);
    // Close it
    sock->Close();
    // Done
    return;
    }
  currMsg->messageNumber += bytes;
  // There is still data to receive
  if (currMsg->messageNumber != currMsg->dataLength)
    {
    // Save message
    RTDS_msgSave(currMsg);
    // Done
    return;
    }
  // Completed
  else
    {
    // Forget socket
    RTDS_remove(sock);
    // Close it
    sock->Close();
    }
  // Unpack data
  uint32_t packetHeader[10];
  memcpy(packetHeader, currMsg->pData, 40);
  currMsg->messageNumber = (int) packetHeader[1];
  currMsg->messageUniqueId = (int) packetHeader[2];
  currMsg->sender.processNumber = (int) packetHeader[3]; 
  currMsg->sender.instanceNumber = (unsigned int) packetHeader[4];
  currMsg->sender.componentNumber = (unsigned int) packetHeader[5];
  currMsg->receiver.processNumber = (int) packetHeader[6];
  currMsg->receiver.instanceNumber = (unsigned int) packetHeader[7];
  currMsg->receiver.componentNumber = (unsigned int) packetHeader[8];
  long dataLength;
  unsigned char *pData;
  dataLength = (long) packetHeader[9];
  pData = (unsigned char *) malloc(dataLength);
  memcpy(pData, currMsg->pData + 40, dataLength);
  // Trace
  unsigned int senderNodeId = RTDS_parentScheduler->component->node->getComponent(currMsg->sender.componentNumber)->node->nodeUniqueId;
  fprintf(RTDS_traceFile, "    <information nId=\"n%u\" cId=\"c%u\" time=\"%"PRId64"\" pName=\"p%d\" pId=\"%u\" message=\"in from %u\" />\n",
    RTDS_parentScheduler->component->node->nodeUniqueId,
    RTDS_parentScheduler->component->componentUniqueId,
    RTDS_TickGet(), 
    RTDS_currentContext->mySdlInstanceId.processNumber, 
    RTDS_currentContext->mySdlInstanceId.instanceNumber, 
    senderNodeId
  );
  fprintf(RTDS_traceFile, "    <packetReceived nId=\"n%u\" cId=\"c%u\" time=\"%"PRId64"\" nSender=\"n%u\" pktName=\"m%d\" pktId=\"%d\" />\n",
		RTDS_parentScheduler->component->node->nodeUniqueId,
    RTDS_parentScheduler->component->componentUniqueId,
    RTDS_TickGet(), 
    senderNodeId, 
    currMsg->messageNumber,
    currMsg->messageUniqueId
  );
  RTDS_ReleaseMessageUniqueId(currMsg->messageUniqueId);
  // Send via scheduler (new message)
  RTDS_parentScheduler->sendMessage(currMsg->sender, currMsg->receiver, currMsg->messageNumber, dataLength, pData);
  // Discard message 
  free(currMsg->pData);
  free(currMsg);
  // Handle message
  ns3::Simulator::ScheduleNow(&RTDS_Scheduler::run, RTDS_parentScheduler);
  }

RTDS_MessageHeader *RTDS_Proxy::RTDS_pending(ns3::Ptr<ns3::Socket> sock)
  {
  // Look for pending message on the socket
  RTDS_MessageHeader *currMsg, *prevMsg = NULL;
  for (currMsg = RTDS_currentContext->readSaveQueue; currMsg != NULL; prevMsg = currMsg, currMsg = currMsg->next)
    {
    if (slots[currMsg->timerUniqueId] == sock)
      {
        if (prevMsg == NULL)
          {
          RTDS_currentContext->readSaveQueue = currMsg->next;
          if (RTDS_currentContext->readSaveQueue == NULL)
            {
            RTDS_currentContext->writeSaveQueue = NULL;
            }
          }
        else
          {
          prevMsg->next = currMsg->next;
          if (RTDS_currentContext->writeSaveQueue == currMsg)
            {
            RTDS_currentContext->writeSaveQueue = prevMsg;
            }
          }
        break;
      }
    }
  return currMsg;
  }

int RTDS_Proxy::RTDS_insert(ns3::Ptr<ns3::Socket> sock)
  {
  // Find empty slot and insert
  int i;
  for (i = 0; i < RTDS_SETSIZE; ++i)
    {
    if (slots[i] == 0)
      {
      slots[i] = sock;
      return i;
      }
    }
  // No more slots
  return 0;
  }
  
void RTDS_Proxy::RTDS_remove(ns3::Ptr<ns3::Socket> sock)
  {
  // Look for slot and reset it
  int i;
  for (i = 0; i < RTDS_SETSIZE; ++i)
    {
    if (slots[i] == sock)
      {
      slots[i] = 0;
      break;
      }
    }
  }
