/* PragmaDev RTDS Nucleus integration */

#include "RTDS_MACRO.h"

/* **************************************************************** *
 * Application_Initialize
 * **************************************************************** *
 * Nucleus PLUS entry point
 * Initialize RTDS environment
 * **************************************************************** *
 * Parameters:
 *        first_available_memory is the pointer to the 
 *        first available memory address
 * Returns:
 *        Nothing
 * **************************************************************** */
void Application_Initialize( void * first_available_memory )
    {
    STATUS status;
    RTDS_RtosTaskId processId = NULL;
    VOID * processStack = NU_NULL;
    
    /* Create a system memory pool that will be used to allocate task stacks, queue areas, etc.  */
    status = NU_Create_Memory_Pool( &System_Memory, "SYSMEM", first_available_memory, SYSTEM_MEMORY_SIZE, 100, NU_FIFO );
    
    /* Check to see if previous operation successful */
    if ( status != NU_SUCCESS )
        {
        /* Call error handling routine */
        ERC_System_Error( status );
        }
    
    processId = ( RTDS_RtosTaskId )RTDS_MALLOC( sizeof( NU_TASK ) );
    processStack = RTDS_MALLOC( RTDS_TASK_STACK_SIZE );

    /* Task creation without auto-start */
    if( NU_Create_Task( processId, "R_Start", RTDS_Start, RTDS_TASK_OPTIONS_NUMBER, NU_NULL, processStack, RTDS_TASK_STACK_SIZE, RTDS_DEFAULT_PROCESS_PRIORITY, 0, NU_PREEMPT, NU_START ) != NU_SUCCESS )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PROCESS_CREATE_TASK );
        }

    }

/* **************************************************************** *
 * RTDS_Malloc
 * **************************************************************** *
 * Allocate memory from the memory pool
 * **************************************************************** *
 * Parameters:
 *        size: the size of the memory to allocate
 * Returns:
 *        pointer to allocated memory if success 
 * **************************************************************** */
VOID * RTDS_Malloc( UNSIGNED size )
    {
    VOID * pointer = NU_NULL;
    
    /* Then we allocate memory with the NU_NO_SUSPEND mode */
    if( NU_Allocate_Memory( &System_Memory, &pointer, size, NU_NO_SUSPEND ) != NU_SUCCESS )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_ALLOCATE_MEMORY_NO_TASK_CONTEXT );
        }
    /* But if pointer is NU_NULL then we don't have enough memory */
    if ( pointer == NU_NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_DONT_HAVE_ENOUGH_MEMORY );
        }
    
    return pointer;
    }
 
/* **************************************************************** *
 * RTDS_Free
 * **************************************************************** *
 * Release a previously allocated memory from the memory pool  
 * **************************************************************** *
 * Parameters: 
 *        ptr: the pointer to deallocate
 * Returns:
 *        Nothing
 * **************************************************************** */
void RTDS_Free( VOID * ptr )
    {
    if( NU_Deallocate_Memory( ptr ) != NU_SUCCESS )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_DEALLOCATE_MEMORY );
        }
    }

/* **************************************************************** *
 * RTDS_GetTimerUniqueId
 * **************************************************************** *
 * Get an available timer unique id.
 * Is used to set timers
 * **************************************************************** *
 * Parameters:
 *       timerList: points to the first element of the chained list 
 *       of RTDS_TimerState
 * Returns:
 *       The unique timer Id
 * **************************************************************** *
 * Context:
 *      The chained list of RTDS_TimerState is sorted by increasing
 *      unique timer id
 *      timerUniqueId = NULL means the message is not a timer; so it
 *      is avoided.
 * **************************************************************** */
long RTDS_GetTimerUniqueId( RTDS_TimerState * timerList )
    {
    RTDS_TimerState * RTDS_prevTimer = NULL;
    RTDS_TimerState * RTDS_timer = NULL;
    long newTimerId;

    RTDS_CRITICAL_SECTION_START;
    /* If list is empty take 1 */
    if ( timerList == NULL )
        {
        newTimerId = 1;
        }
    /* If 1 is available, take it */
    else if ( timerList->timerUniqueId != 1 )
        {
        newTimerId = 1;
        }
    else
        {
        /* If there is a gap in the timerId chained list, take an Id in the gap */
        RTDS_prevTimer = timerList;
        newTimerId = 0;
        for ( RTDS_timer = timerList->next; RTDS_timer != NULL; RTDS_timer = RTDS_timer->next )
            {
            if ( RTDS_timer->timerUniqueId != RTDS_prevTimer->timerUniqueId + 1 ) 
                {
                newTimerId = RTDS_prevTimer->timerUniqueId + 1;
                }
            RTDS_prevTimer = RTDS_timer;
            }
        /* No gap, let's take the next value */
        if ( newTimerId == 0 )
            {
            newTimerId = RTDS_prevTimer->timerUniqueId + 1;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* Check the counter did not go back to 0 */
    if ( newTimerId == 0 )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_NO_MORE_TIMER_UNIQUE_ID );
        }
    
    return newTimerId;
    }

/* **************************************************************** *
 * RTDS_StartTimer
 * **************************************************************** *
 * Starts a watchdog with the necessary parameters to create the
 * timer when it goes off.
 * **************************************************************** *
 * Parameters:
 *        instanceId: instance Id of the receiver
 *        timerNumber: Number of the timer
 *        timerUniqueId: Unique Id of the timer
 *        delay: delay of the timer
 *        timerStateList: Address of the list of timer
 *        currentContext: active task context
 * Returns:
 *        Nothing
 * **************************************************************** */
void RTDS_StartTimer
( RTDS_SdlInstanceId * instanceId,
 long timerNumber,
 long timerUniqueId,
 int delay,
 RTDS_TimerState ** pTimerStateList,
 RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_TimerState * timerState = NULL;
    RTDS_TimerState * previousTimerState = NULL;
    RTDS_TimerState * newTimerState = NULL;

    /* Allocate memory for this structure */
    newTimerState = ( RTDS_TimerState * )RTDS_MALLOC( sizeof( RTDS_TimerState ) );

    /* Allocate memory for the id of the timer control block pointer */
    newTimerState->watchDogId = ( RTDS_TimerId )RTDS_MALLOC( sizeof( NU_TIMER ) );

    /* Initialize the new element */
    newTimerState->state = RTDS_TIMER_OK;
    newTimerState->timerNumber = timerNumber;
    newTimerState->timeoutValue = delay + RTDS_GetSystemTime();
    newTimerState->timerUniqueId = timerUniqueId;
    newTimerState->next = NULL;
    newTimerState->receiverId = instanceId;
    
    /* Allocate memory for the message to be sent by the watchDog function */
    newTimerState->watchDogParam = ( RTDS_MessageHeader * )RTDS_MALLOC( RTDS_QUEUE_MAX_MSG_LENGTH );
    
    /* Fill in the message parameters */
#ifdef RTDS_SIMULATOR
    newTimerState->watchDogParam->messageUniqueId = 0;
#endif
    newTimerState->watchDogParam->messageNumber = timerNumber;
    newTimerState->watchDogParam->timerUniqueId = timerUniqueId;
    newTimerState->watchDogParam->sender = instanceId;
    newTimerState->watchDogParam->receiver = instanceId;
    newTimerState->watchDogParam->dataLength = 0;
    newTimerState->watchDogParam->pData = NULL;
    newTimerState->watchDogParam->next = NULL;
        
    /* Watchdog creation */
    if ( NU_Create_Timer( newTimerState->watchDogId, "WatchDog", RTDS_WatchDogFunction, ( UNSIGNED )newTimerState, delay, 0, NU_ENABLE_TIMER ) != NU_SUCCESS )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_WATCHDOG_START );
        }

   /* Insert the new element */
    if ( *pTimerStateList == NULL ) /* The list is empty */
        {
        *pTimerStateList = newTimerState;
        }
    /* The list is not empty */
    else
        {
        previousTimerState = NULL;
        for ( timerState = *pTimerStateList ; timerState != NULL ; timerState = timerState->next )
            {
            if ( timerState->timerUniqueId > timerUniqueId )
                {
                if ( previousTimerState == NULL )
                    {
                    *pTimerStateList = newTimerState;
                    newTimerState->next = timerState;
                    }
                else
                    {
                    previousTimerState->next = newTimerState;
                    newTimerState->next = timerState;
                    }
                break;
                }
            previousTimerState = timerState;
            }
        /* Inserted at the end of the list */
        if ( timerState == NULL )
            {
            previousTimerState->next = newTimerState;
            }
        }
    /* trace the creation of the timer */
    RTDS_SIMULATOR_TRACE( RTDS_timerStarted, newTimerState, delay, RTDS_currentContext );
    }

/* **************************************************************** *
 * RTDS_WatchDogFunction
 * **************************************************************** *
 * Function called by NUCLEUS when the timer goes off
 * Send a timer message using the parameters given to the watchdog
 * when the timer is started
 * **************************************************************** *
 * Parameters:
 *        argv: Needed parameters to create and send the timer:
 *              - Instance Id of the receiver
 *              - Message for the receiver
 * Returns:
 *        Nothing
 * **************************************************************** */
void RTDS_WatchDogFunction( UNSIGNED argv )
    {
    RTDS_TimerState * parameter = NULL;
    
    /* Retrieve the parameter initialized in RTDS_StartTimer function */
    parameter = ( RTDS_TimerState * )argv;
    
    /* Send the message */
    if ( NU_Send_To_Queue( parameter->receiverId->queueId, ( VOID * )parameter->watchDogParam, RTDS_QUEUE_MAX_MSG_LENGTH_IN_BYTE, NU_NO_SUSPEND ) != NU_SUCCESS )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_Q_SEND_IN_WATCHDOG );
        }
    }

/* **************************************************************** *
 * RTDS_StopTimer
 * **************************************************************** *
 * Stops a timer in trying to delete the watchdog. If unsuccessfull
 * set it cancelled in the timer chained list verified by the RTDS
 * kernel
 * **************************************************************** *
 * Parameters:
 *        timerNumber: number of the timer
 *        pTimerStateList: pointing to the timer chained list
 *        RTDS_currentContext: currentContext of the task
 * Returns:
 *        Nothing
 * **************************************************************** */
void RTDS_StopTimer
( long timerNumber,
 RTDS_TimerState ** pTimerStateList,
 RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_TimerState * RTDS_timer = NULL;
    RTDS_TimerState * RTDS_prevTimer = NULL;
    RTDS_MessageHeader * RTDS_message = NULL;
    RTDS_MessageHeader * RTDS_prevMessage = NULL;

    RTDS_timer = *pTimerStateList;
    
    while( RTDS_timer != NULL )
        {
        if( ( RTDS_timer->timerNumber == timerNumber ) && ( RTDS_timer->state != RTDS_TIMER_CANCELLED ) )
            {
            RTDS_SIMULATOR_TRACE( RTDS_timerCancelled, RTDS_timer, NULL, RTDS_currentContext );
            
            /* Let's try to cancel the watchdog */
            if( NU_Control_Timer( RTDS_timer->watchDogId, NU_DISABLE_TIMER ) == NU_SUCCESS )
                {
                /* Deallocate the parameter of this timer */
                RTDS_FREE( RTDS_timer->watchDogParam );
                RTDS_timer->watchDogParam = NULL;
                /* Delete the timer */
                if( NU_Delete_Timer( RTDS_timer->watchDogId ) != NU_SUCCESS )
                    {
                    RTDS_SYSTEM_ERROR( RTDS_ERROR_DELETE_WATCHDOG );
                    }
                /* Remove it from the list */
                /* Is the first of the list */
                if ( RTDS_prevTimer == NULL )
                    {
                    *pTimerStateList = RTDS_timer->next;
                    }
                /* Not the first of the list */
                else
                    {
                    RTDS_prevTimer->next = RTDS_timer->next;
                    }
                /* Then deallocate timer structure */
                RTDS_FREE( RTDS_timer->watchDogId );
                RTDS_timer->watchDogId = NULL;
                RTDS_FREE( RTDS_timer );
                RTDS_timer = NULL;
                }
            /* Could not cancel the timer. Probably went off allready */
            /* Set it cancelled in the list */
            else
                {
                RTDS_timer->state = RTDS_TIMER_CANCELLED;
                }
            return;
            }
        RTDS_prevTimer = RTDS_timer;
        RTDS_timer = RTDS_timer->next;
        }
    
    /* If execution gets here: the timer might be in the save queue */
    RTDS_prevMessage = NULL;
    for ( RTDS_message = RTDS_currentContext->readSaveQueue ; RTDS_message != NULL ; RTDS_message = RTDS_message->next )
        {
        if ( RTDS_message->messageNumber == timerNumber )
            {
            /* Remove it from the list */
            /* Is the first of the list */
            if ( RTDS_prevMessage == NULL )
                {
                RTDS_currentContext->readSaveQueue = RTDS_message->next;
                }
            /* Not the first of the list */
            else
                {
                RTDS_prevMessage->next = RTDS_message->next;
                }
            /* Deallocate memory of the message */
            RTDS_FREE( RTDS_message );
            RTDS_message = NULL;
            return;
            }
        RTDS_prevMessage = RTDS_message;
        }
    
    /* If execution gets here: the timer might be in the save queue */
    RTDS_prevMessage = NULL;
    for ( RTDS_message = RTDS_currentContext->writeSaveQueue ; RTDS_message != NULL ; RTDS_message = RTDS_message->next )
        {
        if ( RTDS_message->messageNumber == timerNumber )
            {
            /* Remove it from the list */
            /* Is the first of the list */
            if ( RTDS_prevMessage == NULL )
                {
                RTDS_currentContext->writeSaveQueue = RTDS_message->next;
                }
            /* Not the first of the list */
            else
                {
                RTDS_prevMessage->next = RTDS_message->next;
                }
            /* Deallocate memory of the message */
            RTDS_FREE( RTDS_message );
            RTDS_message = NULL;
            return;
            }
        RTDS_prevMessage = RTDS_message;
        }
    }

/* **************************************************************** *
 * RTDS_GetProcessInstanceId
 * **************************************************************** *
 * Returns the instance Id of process based on its number
 * **************************************************************** *
 * Parameters:
 *        processNumber: Number of the process defined in RTDS_gen.h
 * Returns:
 *        the process queue id if found
 * Error:
 *        System error call if not found and NULL returned
 * **************************************************************** */
RTDS_SdlInstanceId * RTDS_GetProcessInstanceId( int processNumber )
    {
    RTDS_GlobalProcessInfo * tmpGlobalProcessInfo = NULL;
    
    tmpGlobalProcessInfo = RTDS_globalProcessInfo;
    
    RTDS_CRITICAL_SECTION_START;
    while ( tmpGlobalProcessInfo != NULL )
        {
        if ( tmpGlobalProcessInfo->sdlProcessNumber == processNumber )
            {
            break;
            }
        tmpGlobalProcessInfo = tmpGlobalProcessInfo->next;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    if ( tmpGlobalProcessInfo != NULL )
        {
        return ( tmpGlobalProcessInfo->mySdlInstanceId );
        }
    else
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_QUEUE_ID );
        return ( NULL );
        }
    }

/* **************************************************************** *
 * RTDS_MsgQueueCreate
 * **************************************************************** *
 * Create a message queue
 * **************************************************************** *
 * Parameters:
 *        Nothing
 * Returns:
 *       The created message queue ID
 * **************************************************************** */
RTDS_RtosQueueId RTDS_MsgQueueCreate( void )
    {
    RTDS_RtosQueueId queueId = NULL;
    VOID * queueArea = NU_NULL;
    
    queueArea = RTDS_MALLOC( RTDS_QUEUE_MAX_MSG * RTDS_QUEUE_MAX_MSG_LENGTH );
    queueId = ( RTDS_RtosQueueId )RTDS_MALLOC( sizeof( NU_QUEUE ) );
    
    /* Create the communication queue */
    if ( NU_Create_Queue( queueId, "NONE", queueArea, RTDS_QUEUE_MAX_MSG * RTDS_QUEUE_MAX_MSG_LENGTH_IN_BYTE, NU_FIXED_SIZE, RTDS_QUEUE_MAX_MSG_LENGTH_IN_BYTE, NU_FIFO ) != NU_SUCCESS )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PROCESS_CREATE_MSG_Q_CREATE );
        }
    
    return queueId;
    }

/* **************************************************************** *
 * RTDS_MsgReceive
 * **************************************************************** *
 * Receives a message from a process instance
 * **************************************************************** *
 * Parameters:
 *        instance id to read the messsage from
 *        pointer on the message to receive
 *        size of message
 *        mode used to read (blocking or non blocking)
 * Returns:
 *       Nothing
 * **************************************************************** */
void RTDS_MsgReceive( RTDS_SdlInstanceId * instanceId, RTDS_MessageHeader ** message )
    {
    /* Receive the message */
    if( RTDS_MsgQueueReceive( instanceId->queueId, message, RTDS_QUEUE_MAX_MSG_LENGTH_IN_BYTE, NU_SUSPEND ) != RTDS_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_RECEIVE );  
        }
    }

/* **************************************************************** *
 * RTDS_MsgReceive
 * **************************************************************** *
 * Receives a message from a process's queue
 * **************************************************************** *
 * Parameters:
 *        queue id to read the messsage from
 *        pointer on the message to receive
 *        size of message
 *        mode used to read (blocking or non blocking)
 * Returns:
 *       RTDS_OK if everything is OK RTDS_ERROR otherwise
 * **************************************************************** */
int RTDS_MsgQueueReceive( RTDS_RtosQueueId queueId, RTDS_MessageHeader ** message, int size, int mode )
    {
    UNSIGNED actual_size;
    
    /* Receive the message */
    if( NU_Receive_From_Queue( queueId, ( VOID * )*message, size, &actual_size, mode ) != NU_SUCCESS )
        {
        return RTDS_ERROR;
        }
    
    return RTDS_OK;
    }

/* **************************************************************** *
 * RTDS_MsgQueueSend
 * **************************************************************** *
 * Send a message in a process's queue
 * **************************************************************** *
 * Parameters:
 *        messageNumber representing a message name
 *        dataLength length of data pointed by pData
 *        pData  pointer on data sent with the message
 *        receiver message receiver queue address
 *        sender message sender queue address
 *        RTDS_currentContext currentContext of the task
 * Returns:
 *       Nothing
 * **************************************************************** */
void RTDS_MsgQueueSend
( long messageNumber, 
 long dataLength, 
 unsigned char * pData, 
 RTDS_SdlInstanceId * receiver ,
 RTDS_SdlInstanceId * sender ,
 RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_MessageHeader * RTDS_messageToSend = NULL;
    
    /* Memory allocation for the message to send */
    RTDS_messageToSend = ( RTDS_MessageHeader * )RTDS_MALLOC( RTDS_QUEUE_MAX_MSG_LENGTH );
    
    /* Fill the message fields */
    RTDS_messageToSend->messageNumber = messageNumber;
    RTDS_messageToSend->timerUniqueId = 0;
    RTDS_messageToSend->sender = sender;
    RTDS_messageToSend->receiver = receiver;
    RTDS_messageToSend->dataLength = dataLength;
    RTDS_messageToSend->pData = pData;
    RTDS_messageToSend->next = NULL;
    
#ifdef RTDS_SIMULATOR
    RTDS_messageToSend->messageUniqueId = RTDS_GetMessageUniqueId();
    RTDS_messageDataToString( &RTDS_globalPrintableParameters, messageNumber, dataLength, ( void * )pData, RTDS_PARAM_CODEC_MAX_DEPTH );
    RTDS_SIMULATOR_TRACE( RTDS_messageSent, RTDS_messageToSend, receiver, RTDS_currentContext );
    RTDS_FREE( RTDS_globalPrintableParameters );
    RTDS_globalPrintableParameters = NULL;
#endif
    /* Send the message */
    if ( NU_Send_To_Queue( receiver->queueId, ( VOID * )RTDS_messageToSend, RTDS_QUEUE_MAX_MSG_LENGTH_IN_BYTE, NU_SUSPEND ) != NU_SUCCESS )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_SEND );
        }
    /* Release previously allocated memory for the message */      
    RTDS_FREE( RTDS_messageToSend );
    RTDS_messageToSend = NULL;
    }

/* *******************************************************************
 * RTDS_TaskCreate
 *-------------------------------------------------------------------
 * Creates a new thread.
 * If any error is encountered RTDS_SYSTEM_ERROR is called and
 * should stop execution.
 *
 * PARAMETERS:
 *     priority : int             Message queue to delete
 *     functionAddress : void*    Adresse of function to execute
 *     newProcessInfo : void*     Parameters for the created thread
 *
 * RETURN:
 *     RTDS_RtosTaskId             ID of the created task 
 ******************************************************************** */
RTDS_RtosTaskId RTDS_TaskCreate( int priority,  VOID ( * functionAddress )( UNSIGNED, VOID * ), void * newProcessInfo )
    {
    RTDS_RtosTaskId processId = NULL;
    VOID * processStack = NU_NULL;
    
    processId = ( RTDS_RtosTaskId )RTDS_MALLOC( sizeof( NU_TASK ) );
    processStack = RTDS_MALLOC( RTDS_TASK_STACK_SIZE );
    
    /* Task creation with auto-start */
    if ( NU_Create_Task( processId, "NONE", functionAddress, RTDS_TASK_OPTIONS_NUMBER, ( VOID * )newProcessInfo, processStack, RTDS_TASK_STACK_SIZE, priority, 0, NU_PREEMPT, NU_START ) != NU_SUCCESS )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PROCESS_CREATE_TASK );
        }
    
    return processId;
    }

/* **************************************************************** *
 * RTDS_ProcessCreate
 * **************************************************************** *
 * Create a new SDL process and add a processInfo struct in the
 * process info chained list
 * **************************************************************** *
 * Parameters:
 *        name of the process
 *        process number
 *        address of the process function
 *        priority of the process
 *        synchronization parameter indicate if the process can start or not
 *        RTDS_currentContext currentContext of the task
 * Returns:
 *       Nothing but updates the OFFSPRING of the caller
 * **************************************************************** */ 
void RTDS_ProcessCreate
( char * processName,
 int processNumber,
 VOID ( * functionAddress )( UNSIGNED, VOID * ),
 int priority,
 short synchronization,
 RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_GlobalProcessInfo * processInfo = NULL;
    RTDS_GlobalProcessInfo * newProcessInfo = NULL;
    RTDS_GlobalProcessInfo * tmpProcessInfo = NULL;
    RTDS_GlobalProcessInfo * previousProcessInfo = NULL;
    RTDS_GlobalProcessInfo * tmpProcessInfoPtr = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    /* Pointer to the RTDS_globalProcessInfo pointer */
    tmpProcessInfo = RTDS_globalProcessInfoToDelete;
    /* Then we can search in this list all terminate process and thus release all memory they hold */ 
    while ( tmpProcessInfo != NULL )
        {
        /* We check if the process wants to die */
        if ( tmpProcessInfo->processKill == RTDS_PROCESS_FINISHED )
            {
            /* Update the process information chained list */
            if ( previousProcessInfo == NULL )
                {
                RTDS_globalProcessInfoToDelete = tmpProcessInfo->next;
                }
            else
                {
                previousProcessInfo->next = tmpProcessInfo->next;
                }

            /* Then we release previously allocated memory (the pointer to structure of the queue id and the queue message area) */
            RTDS_FREE ( tmpProcessInfo->mySdlInstanceId->queueId );
            tmpProcessInfo->mySdlInstanceId->queueId = NULL;
            RTDS_FREE ( tmpProcessInfo->queueArea );
            tmpProcessInfo->queueArea = NULL;
            RTDS_FREE ( tmpProcessInfo->mySdlInstanceId );
            tmpProcessInfo->mySdlInstanceId = NULL;
            
            /* Then we cant delete the task */
            if ( NU_Delete_Task( tmpProcessInfo->myRtosTaskId ) != NU_SUCCESS )
                {
                RTDS_SYSTEM_ERROR( RTDS_ERROR_TASK_DELETE );
                }
            
            /* Finally we can release previously allocated memory for the stack and the task control block */
            /* and the context of the process */
            RTDS_FREE ( tmpProcessInfo->myRtosTaskId );
            tmpProcessInfo->myRtosTaskId = NULL;
            RTDS_FREE ( tmpProcessInfo->processStack );
            tmpProcessInfo->processStack = NULL;
            tmpProcessInfoPtr = tmpProcessInfo;
            tmpProcessInfo = tmpProcessInfo->next;
            RTDS_FREE ( tmpProcessInfoPtr );
            tmpProcessInfoPtr = NULL;
            }
        else
            {
            previousProcessInfo = tmpProcessInfo;
            tmpProcessInfo = tmpProcessInfo->next;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* Allocate a new processInfo structure */
    newProcessInfo = ( RTDS_GlobalProcessInfo * )RTDS_MALLOC( sizeof( RTDS_GlobalProcessInfo ) );
    /* Allocate memory to the instance ID and queue control block of the task */
    newProcessInfo->mySdlInstanceId = ( RTDS_SdlInstanceId * )RTDS_MALLOC( sizeof( RTDS_SdlInstanceId ) );
    newProcessInfo->mySdlInstanceId->queueId = ( RTDS_RtosQueueId )RTDS_MALLOC( sizeof( NU_QUEUE ) );
    /* Allocate memory for the task control block */
    newProcessInfo->myRtosTaskId = ( RTDS_RtosTaskId )RTDS_MALLOC( sizeof( NU_TASK ) );
    /* Allocate memory for the stack of the task */
    newProcessInfo->processStack = RTDS_MALLOC( RTDS_TASK_STACK_SIZE );
    /* Allocate memory to the queue of the task */
    newProcessInfo->queueArea = RTDS_MALLOC( RTDS_QUEUE_MAX_MSG * RTDS_QUEUE_MAX_MSG_LENGTH );
    
    /* Create the communication queue of the task */
    if ( NU_Create_Queue( newProcessInfo->mySdlInstanceId->queueId, processName, newProcessInfo->queueArea, RTDS_QUEUE_MAX_MSG * RTDS_QUEUE_MAX_MSG_LENGTH_IN_BYTE, NU_FIXED_SIZE, RTDS_QUEUE_MAX_MSG_LENGTH_IN_BYTE, NU_FIFO ) != NU_SUCCESS )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PROCESS_CREATE_MSG_Q_CREATE );
        }
    if ( NU_Reset_Queue(newProcessInfo->mySdlInstanceId->queueId) != NU_SUCCESS )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PROCESS_CREATE_MSG_Q_CREATE );
        }
    /* Fill fields which are relevant to the created task */
    newProcessInfo->parentSdlInstanceId = RTDS_currentContext->mySdlInstanceId;
    newProcessInfo->offspringSdlInstanceId = NULL;
    newProcessInfo->sdlProcessNumber = processNumber;
    newProcessInfo->sdlState = 0;
    newProcessInfo->processKill = RTDS_PROCESS_NOT_FINISHED;
    newProcessInfo->next = NULL;
    newProcessInfo->currentMessage = NULL;
    newProcessInfo->timerList = NULL;
    newProcessInfo->readSaveQueue = NULL;
    newProcessInfo->writeSaveQueue = NULL;
    
#ifdef RTDS_SIMULATOR
    newProcessInfo->priority = priority;
#endif
    
    /* Task creation without auto-start */
    if ( NU_Create_Task( newProcessInfo->myRtosTaskId, processName, functionAddress, RTDS_TASK_OPTIONS_NUMBER, ( VOID * )newProcessInfo, newProcessInfo->processStack, RTDS_TASK_STACK_SIZE, priority, 0, NU_PREEMPT, NU_NO_START ) != NU_SUCCESS )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PROCESS_CREATE_TASK );
        }
    
    RTDS_CRITICAL_SECTION_START;
    /* Udpdate the offspring field */
    RTDS_currentContext->offspringSdlInstanceId = newProcessInfo->mySdlInstanceId;
    
    /* Add the process information to the chained list pointed by the RTDS_globalProcessInfo global variable */
    if ( RTDS_globalProcessInfo != NULL )
        {
        processInfo = RTDS_globalProcessInfo;
        while( processInfo->next != NULL )
            {
            processInfo = processInfo->next;
            }
        processInfo->next = newProcessInfo; 
        }
    else
        {
        RTDS_globalProcessInfo = newProcessInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    if ( synchronization == RTDS_HOLD )
        {
        if( NU_Resume_Task( newProcessInfo->myRtosTaskId ) != NU_SUCCESS )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_PROCESS_RESUME );
            }
        }
    
    /* Trace the process creation */
    RTDS_SIMULATOR_TRACE( RTDS_processCreated, newProcessInfo, NULL, RTDS_currentContext );
    }

/* **************************************************************** *
 * RTDS_ProcessForget
 *-------------------------------------------------------------------
 * Forgets a given process, freeing all associated information
 *
 * PARAMETERS:
 *  - RTDS_currentContext : The current context of the task to delete
 * RETURNS:
 *  - The PID for the forgotten process or NULL if process was not found
 * **************************************************************** */
RTDS_RtosTaskId RTDS_ProcessForget( RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_TimerState * RTDS_timer = NULL;
    RTDS_MessageHeader * message = NULL;
    RTDS_MessageHeader * tmpMessage = NULL;
    RTDS_GlobalProcessInfo * tmpProcessInfo = NULL;
    RTDS_GlobalProcessInfo * tmpProcessInfoToDelete = NULL;
    RTDS_GlobalProcessInfo * previousProcessInfo = NULL;   
    RTDS_RtosTaskId processToKill = NULL;
    
    /* Let's free the current message if any */
    if ( RTDS_currentContext->currentMessage != NULL )
        {
        /* Free the message parameter structure if any */
        if (RTDS_currentContext->currentMessage->pData != NULL){
          RTDS_FREE(RTDS_currentContext->currentMessage->pData);
          }

        #ifdef RTDS_SIMULATOR
            /* Release the message unique id back to the pool */
            RTDS_ReleaseMessageUniqueId( RTDS_currentContext->currentMessage->messageUniqueId );
        #endif
        /* Free memory */
        RTDS_FREE( RTDS_currentContext->currentMessage );
        RTDS_currentContext->currentMessage = NULL;
        }
    
    /* Let's clean the timer chained list to free memory and stop watchdogs */
    for ( RTDS_timer = RTDS_currentContext->timerList; RTDS_timer != NULL; RTDS_timer = RTDS_currentContext->timerList )
        {
        /* Disable the timer */
        if ( NU_Control_Timer( RTDS_timer->watchDogId, NU_DISABLE_TIMER ) == NU_SUCCESS )
            {
            /* Deallocate the parameter of this timer */
            RTDS_FREE( RTDS_timer->watchDogParam );
            RTDS_timer->watchDogParam = NULL;
            /* Then delete it */
            if( NU_Delete_Timer( RTDS_timer->watchDogId ) != NU_SUCCESS )
                {
                RTDS_SYSTEM_ERROR( RTDS_ERROR_DELETE_WATCHDOG );
                }
            /* Remove it from the list of timers */
            RTDS_currentContext->timerList = RTDS_timer->next;
            /* Then deallocate timer structure */
            RTDS_FREE( RTDS_timer->watchDogId );
            RTDS_timer->watchDogId = NULL;
            }
        }
    
    /* Clean the save queue: free messages and message unique ids in the read and write save queues */
    for ( message = RTDS_currentContext->readSaveQueue; message != NULL; )
        {
        #ifdef RTDS_SIMULATOR
            RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
        #endif
        tmpMessage = message;
        message = message->next;
        RTDS_FREE( tmpMessage );
        tmpMessage = NULL;
        }
    
    for ( message = RTDS_currentContext->writeSaveQueue; message != NULL; )
        {
        #ifdef RTDS_SIMULATOR
            RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
        #endif
        tmpMessage = message;
        message = message->next;
        RTDS_FREE( tmpMessage );
        tmpMessage = NULL;
        }
    
    /* Discard all messages in the message queue */
    if ( NU_Reset_Queue( RTDS_currentContext->mySdlInstanceId->queueId ) != NU_SUCCESS )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_RESET );
        }  
    /* If yes we delete its queue id */
    if ( NU_Delete_Queue( RTDS_currentContext->mySdlInstanceId->queueId ) != NU_SUCCESS )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_DELETE );
        }
    
    RTDS_CRITICAL_SECTION_START;
    tmpProcessInfo = RTDS_globalProcessInfo;
    /* Then we can search in this list and thus release all memory it hold */ 
    while ( tmpProcessInfo != NULL )
        {
        if ( tmpProcessInfo->mySdlInstanceId == RTDS_currentContext->mySdlInstanceId )
            {
            /* Put the process context in the to delete list */
            if ( RTDS_globalProcessInfoToDelete == NULL )
              RTDS_globalProcessInfoToDelete = tmpProcessInfo;
            else {
              for ( tmpProcessInfoToDelete = RTDS_globalProcessInfoToDelete; tmpProcessInfoToDelete->next!=NULL ; tmpProcessInfoToDelete=tmpProcessInfoToDelete->next ) ;
              tmpProcessInfoToDelete->next = tmpProcessInfo;
              }
            
            /* Update the process information chained list */
            if ( previousProcessInfo == NULL )
                {
                RTDS_globalProcessInfo = tmpProcessInfo->next;
                }
            else
                {
                previousProcessInfo->next = tmpProcessInfo->next;
                }
            
            tmpProcessInfo->next = NULL;
            processToKill = tmpProcessInfo->myRtosTaskId;
            tmpProcessInfo->processKill = RTDS_PROCESS_FINISHED;
            
            /* Trace the process terminaison */
            RTDS_SIMULATOR_TRACE( RTDS_processDied, tmpProcessInfo, NULL, RTDS_currentContext );

            break;
            }
        else
            {
            previousProcessInfo = tmpProcessInfo;
            tmpProcessInfo = tmpProcessInfo->next;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    return processToKill;
    }
/* **************************************************************** *
 * RTDS_ProcessKill
 * ****************************o************************************ *
 * Kills an SDL process and delete its queue and process info block
 * **************************************************************** *
 * Parameters:
 *        the address of the process current context
 * Returns:
 *        Nothing
 * **************************************************************** */
void RTDS_ProcessKill( RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_RtosTaskId pidToKill = NULL;
    
    /* Forget everything about process, delete its message queue and get its PID */
    pidToKill = RTDS_ProcessForget( RTDS_currentContext );
    
    /* Delete the task */
    if ( pidToKill != NULL )
        {
        /* Before deleting the task we have to terminate it execution */
        if ( NU_Terminate_Task( pidToKill ) != NU_SUCCESS )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_TASK_FINISH );
            }
        }
    else
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_TASK_TO_DELETE_NOT_FOUND );
        }
    }

/* **************************************************************** *
 * RTDS_Semaphore_Create
 * **************************************************************** *
 * Create a semaphore
 * **************************************************************** *
 * Parameters:
 *        the name of the semaphore
 *        the option of the created semaphore 
 *        (specifies how tasks suspend on the semaphore by priority or fifo policy)
 *        the initial count of the semaphore
 * Returns:
 *        the semaphore id of the created semaphore
 * **************************************************************** */
RTDS_SemaphoreId RTDS_Semaphore_Create( char * name, OPTION opt, int initialCount )
    {
    RTDS_SemaphoreId sem = NULL;
    
    /* Allocate memory for the semaphore control block */
    sem = ( RTDS_SemaphoreId )RTDS_MALLOC( sizeof( NU_SEMAPHORE ) );
    
    /* Create the semaphore */
    if( NU_Create_Semaphore( sem, name, initialCount, opt ) != NU_SUCCESS )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_CREATE_SEMAPHORE );
        }
    
    /* Return the id of the semaphore */
    return sem;
    }

/* **************************************************************** *
 * RTDS_Sem_Info_Insert
 * **************************************************************** *
 * Inserts a semaphoreInfo struct in the semaphore info chained list
 * **************************************************************** *
 * Parameters:
 *        semaphore number
 *        id of the semaphore
 *        RTDS_currentContext currentContext of the task
 * Returns:
 *        The semaphore id so that the user can use the semaphore id directly
 * **************************************************************** */
RTDS_SemaphoreId RTDS_Sem_Info_Insert( int semaphoreNumber, RTDS_SemaphoreId semaphoreId, RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_GlobalSemaphoreInfo * semInfo = NULL;
    RTDS_GlobalSemaphoreInfo * newSemInfo = NULL;
    
    /* Allocate and fill in a new semInfo structure */
    newSemInfo = ( RTDS_GlobalSemaphoreInfo * )RTDS_MALLOC( sizeof( RTDS_GlobalSemaphoreInfo ) );
    newSemInfo->semaphoreNumber = semaphoreNumber;
    newSemInfo->next = NULL;
    newSemInfo->semaphoreId = semaphoreId;
    
    RTDS_CRITICAL_SECTION_START;
    /* Add the semaphore information to the chained list pointed by the RTDS_globalSemaphoreInfo global variable */
    if ( RTDS_globalSemaphoreInfo == NULL )
        {
        RTDS_globalSemaphoreInfo = newSemInfo;
        }
    else
        {
        /* Let's get to the end of the list */
        for ( semInfo = RTDS_globalSemaphoreInfo; semInfo->next != NULL; semInfo = semInfo->next );
        semInfo->next = newSemInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* Trace the semaphore creation */
    RTDS_SIMULATOR_TRACE( RTDS_semaphoreCreated, semaphoreId, -1, RTDS_currentContext );
    
    /* Return the semaphore id */
    return semaphoreId;
    }

/* **************************************************************** *
 * RTDS_Sem_Delete
 * **************************************************************** *
 * Delete a Nucleus semaphore and delete its info from the semaphore
 * information chained list and free the related memory
 * **************************************************************** *
 * Parameters:
 *        the address of the semaphore to find the semaphore info block
 *        RTDS_currentContext currentContext of the task
 * Returns:
 *        Nothing
 * **************************************************************** */
void RTDS_Sem_Delete( RTDS_SemaphoreId semaphoreId, RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_GlobalSemaphoreInfo * semInfo = NULL;
    RTDS_GlobalSemaphoreInfo * semInfoPrev = NULL;
    
    RTDS_SIMULATOR_TRACE( RTDS_semaphoreDeleted, semaphoreId, 0, RTDS_currentContext );
    
    /* Remove the semaphore information from the chained list */
    semInfoPrev = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    for ( semInfo = RTDS_globalSemaphoreInfo ; semInfo != NULL ; semInfo = semInfo->next )
        {
        /* semaphore found */
        if ( semInfo->semaphoreId == semaphoreId )
            {
            if ( semInfoPrev == NULL )
                {
                RTDS_globalSemaphoreInfo = semInfo->next;
                }
            else
                {
                semInfoPrev->next = semInfo->next;
                }
            /* Delete the semaphore */
            if( NU_Delete_Semaphore( semInfo->semaphoreId ) != NU_SUCCESS )
                {
                RTDS_SYSTEM_ERROR( RTDS_ERROR_SEMAPHORE_DELETE );
                }
            /* Release previously allocated memory */
            RTDS_FREE( semInfo->semaphoreId );
            semInfo->semaphoreId = NULL;
            RTDS_FREE( semInfo );
            semInfo = NULL;
            break;
            }
        semInfoPrev = semInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    }

/* **************************************************************** *
 * RTDS_SemaphoreIdTake
 * **************************************************************** *
 * Attempt to take a semaphore from its id
 * **************************************************************** *
 * Parameters:
 *        the id of the semaphore
 *        timeout value
 *        RTDS_currentContext currentContext of the task
 * Returns:
 *        RTDS_ERROR if the timeout elapsed RTDS_OK else
 * **************************************************************** */
RTDS_SemaphoreStatus RTDS_SemaphoreIdTake( RTDS_SemaphoreId semaphoreId, RTDS_SemaphoreTimeout TIME_OUT, RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_SemaphoreStatus tmpStatus = RTDS_ERROR;
    
    RTDS_SIMULATOR_TRACE( RTDS_semTakeAttempt, semaphoreId, TIME_OUT, RTDS_currentContext );
    
    if ( ( NU_Obtain_Semaphore( semaphoreId, TIME_OUT ) ) == NU_SUCCESS )
        {
        tmpStatus = RTDS_OK;
        RTDS_SIMULATOR_TRACE( RTDS_semTakeSucceded, semaphoreId, -1, RTDS_currentContext );
        }
    else
        {
        RTDS_SIMULATOR_TRACE( RTDS_semTakeTimedOut, semaphoreId, -1, RTDS_currentContext );
        }
    
    return tmpStatus;
    }

/* **************************************************************** *
 * RTDS_SemaphoreIdGive
 * **************************************************************** *
 * Attempt to release a semaphore from its id
 * **************************************************************** *
 * Parameters:
 *        the id of the semaphore
 * Returns:
 *        Nothing
 * **************************************************************** */
void RTDS_SemaphoreIdGive( RTDS_SemaphoreId semaphoreId )
    {
     
    /* Release the semaphore */
    if( NU_Release_Semaphore( semaphoreId ) != NU_SUCCESS )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEMAPHORE_ID_GIVE );
        }
    }

/* **************************************************************** *
 * RTDS_GetSemaphoreId
 * **************************************************************** *
 * Gets the id of a semaphore from its number (name)
 * **************************************************************** *
 * Parameters:
 *        semaphore number representing its name
 * Returns:
 *        the id of the semaphore
 * **************************************************************** */
RTDS_SemaphoreId RTDS_GetSemaphoreId( int semaphoreNumber )
    {
    RTDS_GlobalSemaphoreInfo * semInfo = NULL;
    RTDS_SemaphoreId foundSemaphoreId = NULL;

    RTDS_CRITICAL_SECTION_START;
    for ( semInfo = RTDS_globalSemaphoreInfo; semInfo != NULL; semInfo = semInfo->next )
        {
        /* Semaphore found */
        if ( semInfo->semaphoreNumber == semaphoreNumber )
            {
            foundSemaphoreId = semInfo->semaphoreId;
            break;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* Return the semaphore id */
    return foundSemaphoreId;
    }

/* **************************************************************** *
 * RTDS_TransitionCleanUp
 * **************************************************************** *
 * Called at the end of transitions:
 * - frees message buffer if valid
 * - re-organize save queue if state has changed
 * **************************************************************** *
 * Parameters:
 *      Pointer to the current context of the task
 *      Previous sdl state
 * Returns:
 *      Nothing
 * **************************************************************** */
void RTDS_TransitionCleanUp( RTDS_GlobalProcessInfo * RTDS_currentContext, int RTDS_sdlStatePrev )
    {
    /* Free message buffer if valid */
    if ( RTDS_currentContext->currentMessage != NULL )
        {
        /* Free the message parameter structure if any */
        if (RTDS_currentContext->currentMessage->pData != NULL){
          RTDS_FREE(RTDS_currentContext->currentMessage->pData);
          }
        #ifdef RTDS_SIMULATOR
            /* Release the message unique id back to the pool */
            RTDS_ReleaseMessageUniqueId( RTDS_currentContext->currentMessage->messageUniqueId );
        #endif

        /* Free memory */
        RTDS_FREE( RTDS_currentContext->currentMessage );
        RTDS_currentContext->currentMessage = NULL;
        } /* End of if ( RTDS_currentContext->currentMessage != NULL ) */
    
    /* If SDL state has changed and messages have been saved: reorganise the save queue */
    if ( ( RTDS_currentContext->sdlState != RTDS_sdlStatePrev ) && ( RTDS_currentContext->writeSaveQueue != NULL ) )
        {
        RTDS_MessageHeader * message = NULL;
        /* Let's get to the end of the save queue */
        for ( message = RTDS_currentContext->writeSaveQueue; message->next != NULL; message = message->next );
        message->next = RTDS_currentContext->readSaveQueue;
        RTDS_currentContext->readSaveQueue = RTDS_currentContext->writeSaveQueue;
        RTDS_currentContext->writeSaveQueue = NULL;
        }
    }

/* **************************************************************** *
 * RTDS_GetSystemTime
 * **************************************************************** *
 * As its name states...
 * **************************************************************** *
 * Parameters:
 *      None
 * Returns:
 *      System tick count value
 * **************************************************************** */
unsigned long RTDS_GetSystemTime( void )
    {
    return NU_Retrieve_Clock();
    }

/* **************************************************************** *
 * **************************************************************** *
 * **************************************************************** *
 * THE CODE BELOW IS ONLY USED TO DEBUG WITH RTDS SDL-RT DEBUGGER
 * **************************************************************** *
 * **************************************************************** *
 * **************************************************************** */

#ifdef RTDS_SIMULATOR

unsigned char * RTDS_globalMessageUniqueIdPool = NULL;
unsigned long RTDS_globalSystemTime;
RTDS_GlobalTraceInfo RTDS_globalTraceEntry = { RTDS_systemError, NULL, 0, NULL };
char * RTDS_globalPrintableParameters = NULL;

/* **************************************************************** *
 * RTDS_GetMessageUniqueId
 * **************************************************************** *
 * Gets a message unique id for the simulator
 * **************************************************************** *
 * Parameters:
 *      None
 * Returns:
 *      the message unique id (minimum is 1)
 * **************************************************************** */
unsigned long RTDS_GetMessageUniqueId( void )
    {
    unsigned char * index = NULL;
    long uniqueByteId;
    long uniqueBitId;
    
    index = RTDS_globalMessageUniqueIdPool;
    
    RTDS_CRITICAL_SECTION_START;
    for ( uniqueByteId = 0; uniqueByteId < RTDS_MESSAGE_UNIQUE_ID_POOL_SIZE; uniqueByteId++ )
        {
        if ( *index != 0xFF )
            {
            for ( uniqueBitId = 0; uniqueBitId < 8; uniqueBitId++ )
                {
                if ( ( ( 1 << uniqueBitId ) & *index ) == 0 )
                    {
                    *index = *index | ( 1 << uniqueBitId );
                    RTDS_CRITICAL_SECTION_STOP;
                    return ( 8 * uniqueByteId + uniqueBitId + 1 );
                    }
                }
            }
        index++;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* All bits are set... No more message unique id */
    RTDS_SYSTEM_ERROR( RTDS_ERROR_NO_MORE_MSG_UNIQUE_ID );
    
    return 0;
    }

/* **************************************************************** *
 * RTDS_ReleaseMessageUniqueId
 * **************************************************************** *
 * Make a message unique id available from the pool
 * **************************************************************** *
 * Parameters:
 *      message unique id
 * Returns:
 *      Nothing
 * **************************************************************** */
void RTDS_ReleaseMessageUniqueId( unsigned long messageUniqueId )
    {
    unsigned char * index = NULL;
    
    /* probably a timer */
    if ( messageUniqueId == 0 )
        {
        return;
        }
    messageUniqueId -= 1;
    index = RTDS_globalMessageUniqueIdPool;
    index += ( unsigned char )( messageUniqueId / 8 );
    
    RTDS_CRITICAL_SECTION_START;
    ( *index ) = ( *index ) & ~( 1 << messageUniqueId % 8 );
    RTDS_CRITICAL_SECTION_STOP;
    
    }

/* **************************************************************** *
 * RTDS_DummyTraceFunction
 * **************************************************************** *
 * As its name states... The simulator sets a breakpoint on this
 * function and reads the RTDS_globalTraceEntry variable to see
 * what happened
 * **************************************************************** *
 * Parameters:
 *      None
 * Returns:
 *      Nothing
 * **************************************************************** */
void RTDS_DummyTraceFunction( void )
    {
    }

/* **************************************************************** *
 * RTDS_GetActiveThread
 * **************************************************************** *
 * The simulator call this function and reads the returned value
 * thus he know the id of current task
 * **************************************************************** *
 * Parameters:
 *      None
 * Returns:
 *      The id of the current task
 * **************************************************************** */
NU_TASK * RTDS_GetActiveThread( void )
    {
    return NU_Current_Task_Pointer();
    }

/* * **************************************************************** *
 * RTDS_SimulatorMsgQueueSend
 * **************************************************************** *
 * Send a message in a process's queue from the simulator
 * Basically the same as RTDS_MsgQueueSend but it gets the
 * MsgUniqueId as a parameter and does not call the dummy trace
 * function
 * **************************************************************** *
 * Parameters:
 *        messageNumber representing a message name
 *        dataLength length of data pointed by pData
 *        pData pointer on data sent with the message
 *        receiver message receiver queue address
 *        sender message sender queue address
 *        MsgUniqueId unique id of the message
 * Returns:
 *        Nothing
 * *****************************************************************/
void RTDS_SimulatorMsgQueueSend
( long messageNumber,
 long dataLength,
 unsigned char * pData,
 RTDS_SdlInstanceId * receiver,
 RTDS_SdlInstanceId * sender,
 long MsgUniqueId )
    {
    RTDS_MessageHeader RTDS_messageToSend;
    
    RTDS_messageToSend.messageNumber = messageNumber;
    RTDS_messageToSend.timerUniqueId = 0;
    RTDS_messageToSend.sender = sender;
    
    if ( dataLength != 0 )
        {
        RTDS_stringToMessageData( messageNumber, &( RTDS_messageToSend.dataLength ), ( void ** )&( RTDS_messageToSend.pData ), ( char * )pData );
        RTDS_FREE( pData );
        pData = NULL;
        }
    else
        {
        RTDS_messageToSend.dataLength = 0;
        RTDS_messageToSend.pData = NULL;
        }
    
    RTDS_messageToSend.next = NULL;
    RTDS_messageToSend.messageUniqueId = MsgUniqueId;
    
    /* Specific for the backTrace since the original trace comes from the SDL-RT debugger */
#ifdef RTDS_BACK_TRACE_MAX_EVENT_NUM
    RTDS_globalTraceEntry.event = RTDS_messageSent;
    RTDS_globalTraceEntry.eventParameter1 = ( void * )&RTDS_messageToSend;
    RTDS_globalTraceEntry.eventParameter2 = ( long )receiver;
    RTDS_globalTraceEntry.currentContext = ( RTDS_GlobalProcessInfo * )NULL;
    RTDS_TraceAdd();
#endif
    /* send the message to receiver */
    if ( NU_Send_To_Queue( receiver->queueId, ( VOID * )&RTDS_messageToSend, RTDS_QUEUE_MAX_MSG_LENGTH_IN_BYTE, NU_NO_SUSPEND ) != NU_SUCCESS )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_SEND );
        }
    }

#endif

