#ifndef _RTDS_BASICTYPES_H_
#define _RTDS_BASICTYPES_H_

#ifdef __cplusplus
extern "C" {
#endif

/* OSE includes */
#include "ose.h"
#include "ose_i.h"
#include "tosv.h"
#include "dbgprintf.h"

#include <stdlib.h>
#include <stdarg.h>

#include "outfmt.h"
#include "ose_err.h"

#include "heapapi.h"
#include "heap_err.h"

/*
 * DEFINE FOR SYSTEM OK/ERROR MESSAGE
 * ------------------------------
 */
#define RTDS_OK                             1
#define RTDS_ERROR                          0

/*
 * BASIC DEFINES:
 * ------------------------------
 */
#define RTDS_TRUE 1
#define RTDS_FALSE 0

/*
 * DEFINE FOR QUEUE MANIPULATION:
 * ------------------------------
 */
#define RTDS_QUEUE_MAX_MSG                  20                              /* Maximum number of messages in a queue */
#define RTDS_QUEUE_MAX_MSG_LENGTH           sizeof( RTDS_MessageHeader )    /* Maximum length of messages in a queue */

/*
 * DEFINE FOR SEMAPHORE MANIPULATION:
 * ------------------------------
 */
#define RTDS_SEMAPHORE_TIME_OUT_FOREVER     -1                              /* Time out option when taking the semaphore : wait forever */
#define RTDS_SEMAPHORE_TIME_OUT_NO_WAIT     0                               /* Time out option when taking the semaphore : no wait */

#define RTDS_SEMAPHORE_OPTION_PRIO          2                               /* Semaphore creation option: queuing based on priority */
#define RTDS_SEMAPHORE_OPTION_FIFO          3                               /* Semaphore creation option: queuing based on FIFO */

#define RTDS_BINARY_SEM_INITIAL_EMPTY       0                               /* Binary semaphore creation initial state: empty */
#define RTDS_BINARY_SEM_INITIAL_FULL        1                               /* Binary semaphore creation initial state: full */

#define RTDS_MUTEX_SEM_DELETE_SAFE          6                               /* Protects a task that owns the semaphore from unexpected deletion */
#define RTDS_MUTEX_SEM_INVERSION_SAFE       7                               /* Protect the system from priority inversion */

/*
 * DEFINE FOR PROCESS CREATION:
 * ----------------------------
 */
#define RTDS_HOLD                           1                               /* The newly created task will held at startup on the startup semaphore */
#define RTDS_NO_HOLD                        0                               /* The newly created task will not held at startup on the startup semaphore */
#define RTDS_TASK_OPTIONS                   0                               /* TaskSpawn options */
#define RTDS_TASK_STACK_SIZE                2048                            /* Stack size when creating a new task */
#define RTDS_DEFAULT_PROCESS_PRIORITY       15                              /* If priority not defined in SDL-RT, it will be given this value */

/* ************************************************************************************ *
 * CRITICAL SECTION MACROS
 * ************************************************************************************ */
#define RTDS_CRITICAL_SECTION_DECL          RTDS_SemaphoreId RTDS_globalSystemSemId
#define RTDS_CRITICAL_SECTION_PROTO         extern RTDS_SemaphoreId RTDS_globalSystemSemId
#define RTDS_CRITICAL_SECTION_INIT          RTDS_globalSystemSemId = RTDS_Semaphore_Create( 1, SEM_BINARY )
#define RTDS_CRITICAL_SECTION_START         RTDS_SemaphoreTake( RTDS_globalSystemSemId, RTDS_SEMAPHORE_TIME_OUT_FOREVER )
#define RTDS_CRITICAL_SECTION_STOP          RTDS_SemaphoreIdGive( RTDS_globalSystemSemId )

/* ************************************************************************************ *
 * COVERAGE INFORMATION
 * ************************************************************************************ *
 * The coverage information is stored in an array of unsigned char. The SDL-RT symbol
 * number is used as an index; the value stored is the number of times the symbol has
 * been executed. It can not exceed 0xFF for each symbol.
 * ************************************************************************************ */
#ifdef RTDS_COVERAGE_NB_SYMBOLS
#define RTDS_COVERAGE_DECL                  unsigned char RTDS_globalCoverage[ RTDS_COVERAGE_NB_SYMBOLS ]
#define RTDS_COVERAGE_PROTO                 extern RTDS_COVERAGE_DECL
#define RTDS_COVERAGE_INIT \
    { \
    int i; \
    for( i = 0; i < RTDS_COVERAGE_NB_SYMBOLS; i++ ) \
        { \
        RTDS_globalCoverage[ i ] = 0; \
        } \
    }
#define RTDS_COVERAGE_LOG( SYMBOL_NUMBER ) \
    if( RTDS_globalCoverage[ SYMBOL_NUMBER ] != 0xFF ) \
        { \
        RTDS_globalCoverage[ SYMBOL_NUMBER ]++; \
        }
#else
#define RTDS_COVERAGE_DECL
#define RTDS_COVERAGE_PROTO
#define RTDS_COVERAGE_INIT
#endif

/*
 * DEFINE FOR SIMULATION:
 * ------------------------------
 * Define on how many bytes the messageUniqueId pool will be made of
 * 1 means 1 byte means 1*8 bits means 8 possible simultaneous values
 * Maximum is 8192 since the id a long type
 */
#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER ) || defined( RTDS_FORMAT_TRACE )
    
    #define RTDS_MESSAGE_UNIQUE_ID_POOL_SIZE    64
    
    /* Defines when tracing through socket to synchronize target execution and host debugger */
    #define RTDS_DTRACE_ACK_NOWAIT              0                           /* No acknowledgment from the host */
    #define RTDS_DTRACE_ACK_WAIT                1                           /* Ack from the host */

#ifndef RTDS_PARAM_CODEC_MAX_DEPTH
    #define RTDS_PARAM_CODEC_MAX_DEPTH          4                           /* Number of levels when formatting message parameters to be printable in the SDL-RT debugger */
#endif
    
#ifndef RTDS_PARAM_CODEC_CHUNK_SIZE
    #define RTDS_PARAM_CODEC_CHUNK_SIZE         128                         /* Min size of memory chunk used when formatting message parameters to be printable in the SDL-RT debugger */
#endif

#endif

/*
 * TYPE RTDS_QUEUE.ID:
 * ----------------------
 * Type for a message queue identifier
 * In OSE it is actually the pid
 */
typedef PROCESS RTDS_RtosQueueId;

/*
 * TYPE RTDS_PROCESS.ID:
 * ----------------------
 * Type for a process identifier
 * used to be int in the previous release
 */
typedef PROCESS RTDS_RtosTaskId;

/*
 * TYPE RTDS_SEMAPHORE.ID:
 * ----------------------
 * Type for a semaphore identifier
 */
typedef enum{ SEM_COUNTING, SEM_BINARY, SEM_MUTEX }RTDS_SemaphoreType;

typedef union SemType
    {
    SEMAPHORE * semaphore;        /* Semaphore type */
    MUTEX * mutex;                /* mutex type*/
    }SemType;

typedef struct RTDS_SemType
    {
    RTDS_SemaphoreType typeName;
    SemType semId;
    }RTDS_SemType;

typedef RTDS_SemType * RTDS_SemaphoreId;

/*
 * TYPE RTDS_SEMAPHORE_STATUS:
 * ----------------------
 * Type for a semaphore status
 */
typedef short RTDS_SemaphoreStatus;

/*
 * TYPE RTDS_SEMAPHORE_TIMEOUT:
 * ----------------------
 * Type for a semaphore timeout value
 */
typedef int RTDS_SemaphoreTimeout;

/*
 * TYPE RTDS_PROCESS_PRIORITY:
 * ----------------------
 * Type for process priority
 */
typedef int RTDS_ProcessPriority;

/* In OSE, all signals are of type SIGNAL. By declaring it as a union between
our different signal types, we get a very generic SIGNAL type that
always is treated the same way regardless of contents */

/* typedef RTDS_MessageHeader SIGNAL; */
typedef union SIGNAL
    {
    SIGSELECT sigNo;
    struct RTDS_MessageHeader * messageHeader;
    } SIGNAL;

/*
 * STRUCT RTDS_TIMER.STATE:
 * ------------------------
 * Element in chained list for all active timers
 */
#define RTDS_TIMER_OK           1
#define RTDS_TIMER_CANCELLED    0

typedef struct RTDS_TimerState
    {
    long                        timerNumber;            /* Message number for the timer name */
    long                        timerUniqueId;          /* Timer's unique identifier among all timers */
    unsigned long               timeoutValue;           /* System tick counter value when it will go off */
    struct RTDS_SdlInstanceId   * receiverQueueId;      /* The message queue of the receiver */
    int                         state;                  /* State for the timer (RTDS_TIMER_OK or RTDS_TIMER_CANCELLED) */
    CANCEL_INFO                 watchDogId;             /* Id of the watchDog */
    struct RTDS_TimerState      * next;                 /* 0x18 Next timer in list */
    } RTDS_TimerState;

/*
 * STRUCT RTDS_GLOBAL.SEMAPHORE.INFO:
 * -------------------------------
 * Type for the list of semaphores in the system with their information
 */
typedef struct RTDS_GlobalSemaphoreInfo
    {
    RTDS_SemaphoreId                semaphoreId;        /* The semaphore address */
    int                             semaphoreNumber;    /* Semaphore number representing its name */
    struct RTDS_GlobalSemaphoreInfo * next;             /* Next semaphoreInfo */
    } RTDS_GlobalSemaphoreInfo;

/*
 * STRUCT RTDS_GLOBAL.TRACE.INFO:
 * -------------------------------
 * Structure containing the event information red by the SDL-RT debugger
 */
#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER ) || defined( RTDS_FORMAT_TRACE )
/* Global variable pointing to decoded parmaters */
extern char * RTDS_globalPrintableParameters;
#endif

/*
 * Additional fields in common types
 * ---------------------------------
 */
/* Message header: none */
#define RTDS_MESSAGE_HEADER_ADDITIONNAL_FIELDS \
    SIGSELECT sigNo;                                /* OSE specific compulsory field */

/* Process information: add priority */
#define RTDS_GLOBAL_PROCESS_INFO_ADDITIONNAL_FIELDS \
    RTDS_ProcessPriority priority;                  /* Priority of the process. Only used when debugging. */

#ifdef __cplusplus
}
#endif

#endif /* _RTDS_BASICTYPES_H_ */
