#ifndef _RTDS_BASIC_TYPES_H_
#define _RTDS_BASIC_TYPES_H_

#include <semaphore.h>

/* ERROR MESSAGE */
#define POSIX_SEMAPHORE_LOCKED        0 /* Locked parameter for semaphore creation */
#define POSIX_SEMAPHORE_INTERPROCESS  1 /* Creation parameter for using this sem. in various process */
#define POSIX_SEMAPHORE_MONOPROCESS   0 /* Creation parameter for using this sem. only in the creating process */

/*
*******************************************************************
** Timers should be as prioritary as possible
*******************************************************************
*/
#define RTDS_TIMER_PRIORITY (sched_get_priority_max(SCHED_RR) - 1)

/* POSIX THREAD CREATION PARAMETER: */
#define RTDS_POSIX_TIMER_TASK_OPTIONS   0              /* taskSpawn options */
#define RTDS_POSIX_TIMER_STACK_SIZE     5000           /* Not used : Stack size when creating a new task */

/*****************************************
 *    System defines: OK/ERROR/NONE     *
 *****************************************/
#define RTDS_ERROR                      -1
#define RTDS_OK                          0
#define RTDS_TIMEOUT                     2             /* timeout return value (ususally for semaphores */
#define RTDS_NONE                        0             /* Default option parameter */

/*********************************************
 *    Define for RTDS_OS and RDTS_MACRO     *
 *********************************************/
#define SEM_EMPTY                            0                     /* used for counting and binary */
#define SEM_FULL                             RTDS_NONE             /* used only for binary  */
#define NO_WAIT                              RTDS_NONE             /* Not used */
#define MSG_PRI_NORMAL                       RTDS_NONE             /* Not used */
#define RTDS_SEMAPHORE_MAX_NAME_LENGTH       20                    /* Maximum length of a semaphore name: No restriction for POSIX */
#define RTDS_TASK_OPTIONS                    RTDS_NONE             /* Thread creation options */
#define RTDS_TASK_STACK_SIZE                 5000                  /* Stack size when creating a new task */
#define SEM_C_DEFAULT_CREATION               RTDS_SEM_INTERPROCESS /* This value define the default creation option and can have different meaning depending of the OS */

/********************************************************
 * Define for SDL Symbols (generated by RTDS generator) *
 ********************************************************/
#define RTDS_SEMAPHORE_TIME_OUT_FOREVER      -1                    /* Time out option when taking the semaphore : wait forever */
#define RTDS_SEMAPHORE_TIME_OUT_NO_WAIT      0                     /* Time out option when taking the semaphore : no wait */
#define RTDS_SEMAPHORE_OPTION_PRIO           RTDS_NONE             /* Semaphore creation option: queuing based on priority */
#define RTDS_SEMAPHORE_OPTION_FIFO           RTDS_NONE             /* Semaphore creation option: queuing based on FIFO */
#define RTDS_BINARY_SEM_INITIAL_EMPTY        0                     /* Binary semaphore creation initial state: empty */
#define RTDS_BINARY_SEM_INITIAL_FULL         1                     /* Binary semaphore creation initial state: full */
#define RTDS_MUTEX_SEM_DELETE_SAFE           RTDS_NONE             /* Protects a task that owns the semaphore from unexpected deletion */
#define RTDS_MUTEX_SEM_INVERSION_SAFE        RTDS_NONE             /* Protect the system from priority inversion */
#define RTDS_SEM_INTERPROCESS                1                     /* Creation parameter for using this sem. in various process */
#define RTDS_SEM_MONOPROCESS                 0                     /* Creation parameter for using this sem. only in the creating process */

/*********************************************************************
 * Possible values for Posix/Pthead thread priority depend
 * of the OS (Linux => 0<pri<99  and Solaris => 0<prio<59)
 * higher value correspond to higher priority.
 * Timer thread priority is define in RTDS_OS.h
 * Default is RTDS_TIMER_PRIORITY
 * 98 for Linux and
 * 59 for Solaris and others
 *********************************************************************/

#ifdef __linux__
    /* By default priority is not active since superuser privilege is required */
    /* define  RTDS_THREAD_PRIO_ENABLE if you want to enable priority */
    #define RTDS_DEFAULT_PROCESS_PRIORITY    50
    #define RTDS_DEFAULT_TCP_CLIENT_PRIORITY 98
#else
    #define RTDS_THREAD_PRIO_ENABLE
    #define RTDS_DEFAULT_PROCESS_PRIORITY    20
    #define RTDS_DEFAULT_TCP_CLIENT_PRIORITY 59
#endif

/******************************************************************
 *        Posix task Id
 * ----------------------------------------------------------------
 * SDL task are defined as Posix thread (pthread library)
 ******************************************************************/
typedef pthread_t  RTDS_RtosTaskId;

/******************************************************************
 *        SEMAPHORE TYPES
 * ----------------------------------------------------------------
 *    Use of a UNION to detect whether pthread libray or semaphore
 *    library should be use depending of the semaphore type
 *
 *******************************************************************/
typedef enum  { COUNTING , BINARY , MUTEX } RTDS_TypeName;

typedef union SemType
  {
  sem_t           stdlib_SemType;     /* Semaphore type use in standard posix semaphore lib */
  pthread_mutex_t posixlib_SemType;   /* pthread mutex semaphore type*/
  } SemType;

typedef struct RTDS_SemType
  {
  RTDS_TypeName typeName;
  SemType       semId;
  } RTDS_SemType;


/* RTDS_SemId semaphore type is used in RTDS_OS.c */
typedef RTDS_SemType *RTDS_SemId;

/*
 * TYPE RTDS_SEMAPHORE.ID:
 * ----------------------
 * Type for a semaphore identifier
 */
typedef RTDS_SemId RTDS_SemaphoreId;

/*
 * TYPE RTDS_SEMAPHORE_STATUS:
 * ----------------------
 * Type for a semaphore status
 */
typedef int RTDS_SemaphoreStatus;

/*************************************************
 *        Posix RTDS_SemTimeoutType
 * -----------------------------------------------
 * Type for time to wait for semaphore
 *                (time in ms)
 *************************************************/
typedef long RTDS_SemTimeoutType;

/*
 * TYPE RTDS_SEMAPHORE_TIMEOUT:
 * ----------------------
 * Type for a semaphore timeout value
 */
typedef RTDS_SemTimeoutType RTDS_SemaphoreTimeout;


/*
 * STRUCT RTDS_GLOBAL.SEMAPHORE.INFO:
 * -------------------------------
 * Type for the list of semaphores in the system with their information
 */
typedef struct RTDS_GlobalSemaphoreInfo
    {
    RTDS_SemaphoreId                semaphoreId;     /* The semaphore address  */
    int                             semaphoreNumber; /* Semaphore number representing its name */
    struct RTDS_GlobalSemaphoreInfo *next;           /* Next semaphoreInfo */
    } RTDS_GlobalSemaphoreInfo;


/*
 * TYPE RTDS_PROCESS.PRIORITY.
 * ----------------------
 * Type for process priority
 */
typedef int RTDS_ProcessPriority;


/*
 * STRUCT RTDS_QCB : Queue Control Block
 * -------------------------------------
 * Structure that contains :
 *     - pointer on chain list Semaphore (that protect chained list)
 *     - pointer on the first message
 *     - pointer on Blocking Semaphore (That will allows to wait for a message)
 *
 */
typedef struct RTDS_QCB
  {
  RTDS_SemType                * chainedListSemId; /* Chained list semaphore Id*/
  RTDS_SemType                * blockingSemId;    /* Blocking semaphore Id*/
  struct  RTDS_MessageHeader  * queue;            /* Pointer on the first message in the queue*/
  } RTDS_QCB;


/*
 * TYPE RTDS_RtosQueueId:
 * ---------------------
 * Type for a message queue identifier
 */
typedef RTDS_QCB * RTDS_RtosQueueId;


/*
 * STRUCT RTDS_TIMER.STATE:
 * ------------------------
 * Element in chained list for all active timers
 */
typedef struct RTDS_TimerState
  {
  long                         timerNumber;    /* Message number for the timer name */
  long                         timerUniqueId;  /* Timer's unique identifier among all timers */
  unsigned long                timeoutValue;   /* System tick counter value when it will go off */
  struct RTDS_SdlInstanceId  * receiverId;     /* Instance identifier for the receiver */
  int                          state;          /* State for the timer (RTDS_TIMER_OK or RTDS_TIMER_CANCELLED) */
  RTDS_RtosTaskId              watchDogId;     /* Id of the watchDog */
  struct RTDS_TimerState     * next;           /* Next timer in list */
  } RTDS_TimerState;


/*
 * Additional fields in common types
 * ---------------------------------
 */
/* Message header: none */
#define RTDS_MESSAGE_HEADER_ADDITIONNAL_FIELDS
/* Process information: add priority */
#define RTDS_GLOBAL_PROCESS_INFO_ADDITIONNAL_FIELDS \
  RTDS_ProcessPriority priority;
  
#endif /* _RTDS_BASIC_TYPES_H_ */

