/* PragmaDev RTDS posix integration */

/*
** For some reason, the standard constants for a mutex type are not
** defined by default in pthread.h unless the following #define is
** done...
*/
#undef _GNU_SOURCE
#define _GNU_SOURCE

/* Std include */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <string.h>
#include <pthread.h>

/* Includes to use socket */
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

/* RTDS include */
#include "RTDS_Error.h"
#include "RTDS_OS.h"
#include "RTDS_MACRO.h"

#ifdef RTDS_FORMAT_TRACE
    #include "RTDS_FormatTrace.h"
#endif

#ifdef RTDS_BACK_TRACE_MAX_EVENT_NUM
    #include "RTDS_BackTrace.h"
#endif


#ifdef RTDS_SOCKET_PORT

/********************************************************************
 * RTDS_InitSocket
 *-------------------------------------------------------------------
 * Initialize the connection to the SDL-RT debbuger
 *
 * PARAMETERS
 *    hostNameString : char*    Server IP address
 *    portNum : int             Number of the port
 *
 * RETURN
 *    RTDS_SOCKET_ID_TYPE    
 *********************************************************************/
RTDS_SOCKET_ID_TYPE RTDS_InitSocket( char *hostNameString , int portNum )
    {
    int sd;
    struct sockaddr_in localAddr;
    struct sockaddr_in servAddr;

    /* Create socket */
    sd = socket( AF_INET , SOCK_STREAM , 0 );

    if( sd < 0 )
        {
        /* Cannot open socket */
        return RTDS_ERROR;
        }
        
    /* Bind any port number */
    memset( &localAddr , 0 , sizeof( localAddr ) );
    localAddr.sin_family = AF_INET;
    localAddr.sin_addr.s_addr = htonl( INADDR_ANY );
    localAddr.sin_port = htons( 0 );

    if( bind( sd , ( struct sockaddr * ) &localAddr , sizeof( localAddr ) ) != RTDS_OK )
        {
        /* Cannot bind port */
        return RTDS_ERROR;
        }

    /* Connect to the server */
    memset( &servAddr , 0 , sizeof( servAddr ) );
    servAddr.sin_family = AF_INET;
    servAddr.sin_addr.s_addr = inet_addr( hostNameString );  /* Server IP address */
    servAddr.sin_port = htons( portNum );

    if( connect( sd , ( struct sockaddr * )&servAddr, sizeof( servAddr ) ) != RTDS_OK )
        {
        /* Error didn't connect to server */
        return RTDS_ERROR;
        }

    return sd;
    }

/********************************************************************
 * RTDS_CloseSocket
 *-------------------------------------------------------------------
 * Close a given socket 
 *
 * PARAMETERS
 *     socketId : RTDS_SOCKET_ID_TYPE    Socket ID to close
 *********************************************************************/
void RTDS_CloseSocket( RTDS_SOCKET_ID_TYPE socketId )
    {
    if( close( socketId ) != RTDS_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_CLOSE_SOCKET )
        }
    }

/********************************************************************
 * RTDS_ReadSocket
 *-------------------------------------------------------------------
 * Return a valid data (ended by END_STRING) or RTDS_ERROR
 * if socket connection and reading probleme. This version of function
 * read and process char by char and doesn't use any buffer to read a 
 * certain amount of data before processing.
 *
 * PARAMETERS
 *    socketDescriptor : RTDS_SOCKET_ID_TYPE     Socket ID
 *    data : char**                              Pointer on the value to return
 *    delimiterChar : char                       Delimiter char
 *    escapeChar : char                          Escape char
 *    dataSize : int*                            Size of the message readed
 *                           
 * RETURN
 *    int  CurrentSize or -1 if failing
 *********************************************************************/
int RTDS_ReadSocket( 
 RTDS_SOCKET_ID_TYPE socketDescriptor ,
 char **data ,
 int *dataSize ,
 char delimiterChar ,
 char escapeChar )
    {
    static int n;
    int offset = 0;
    char *tmpReadBuffer;
    int blockSize = 1024;
    int blockFreeSpace;     /* Used to extend memory chunk if needed */
    int nbBlock = 1;        /* Used to count the number of block allocated for the read buffer */
    char tmpchar;
    int escapeFlag=0;       /* indicate that the previous received char was the escapeChar*/

    /* Init the block free space to the initial size */
    blockFreeSpace = blockSize;
    tmpReadBuffer = ( char * )RTDS_MALLOC( blockSize );
    *data = NULL;

    while( 1 )
        {
        n = recv( socketDescriptor , &tmpchar , sizeof( char ) , 0 );
        if ( n < 0 )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_RECV )
            return -1;
            }
        else if (n == 0)
            {
            RTDS_CloseSocket( socketDescriptor );
            return -1;
            }
        else if ( n == 1 )
            {
            if( escapeFlag == 1 )
                {
                if ( tmpchar == delimiterChar )
                    {
                    /* escapeChar was followed by delimiterChar */
                    *dataSize = offset + 2;
                    tmpReadBuffer[ offset ] = tmpchar;
                    *data = tmpReadBuffer;
                    break;
                    }
                else
                    {
                    /* escapeChar was not followed by delimiterChar: wait for escapeChar */
                    escapeFlag = 0;
                    }
                }
            else
                {
                /* waiting for escapeChar */
                if ( tmpchar == escapeChar )
                    {
                    escapeFlag = 1;
                    }
                }
            /* Add char and increment offset */
            tmpReadBuffer[ offset ] = tmpchar;
            offset++;

            if ( !blockFreeSpace )
                {
                nbBlock++;
                tmpReadBuffer = (char *)realloc(tmpReadBuffer, nbBlock * blockSize);
                if (tmpReadBuffer == NULL)
                    {
                    RTDS_SYSTEM_ERROR( RTDS_ERROR_EXTEND_MEMORY_ALLOC )
                    }
                blockFreeSpace = blockSize;
                }
            else
                {
                blockFreeSpace--;
                }
            }/* end of else if n==1 */
        } /* end of while */
    return offset;
    }

/********************************************************************
 * RTDS_SendSocket
 *-------------------------------------------------------------------
 * Send data over the socket
 *
 * PARAMETERS
 *    socketId : RTDS_SOCKET_ID_TYPE    Socket ID
 *    commandString : char*             Pointer on data to send
 *    size : int                        Size to send
 *********************************************************************/
int RTDS_SendSocket( RTDS_SOCKET_ID_TYPE socketId , char *commandString , int size )
    {    
    if ( send( socketId , commandString, ( size_t )size , 0 ) != size )
        {
        return RTDS_ERROR;
        }
    return RTDS_OK;
    }
    
#endif /* RTDS_SOCKET_PORT */


/********************************************************************
 * RTDS_SemMCreate
 *-------------------------------------------------------------------
 * This function creates a mutex semaphore
 *
 * RETURN
 *     Pointer on the created semaphore: SemaphoreId
 *********************************************************************/
RTDS_SemType *RTDS_SemMCreate()
    {
    RTDS_SemType        * SemId = NULL;
    pthread_mutexattr_t   sem_attributes;

    SemId = ( RTDS_SemType * )RTDS_MALLOC( sizeof( RTDS_SemType ) );
    SemId->typeName = MUTEX;

    /* Create recursive mutex semaphore */
    pthread_mutexattr_init(&sem_attributes);
    pthread_mutexattr_settype(&sem_attributes, PTHREAD_MUTEX_RECURSIVE);
    pthread_mutex_init(&(SemId->semId.posixlib_SemType), &sem_attributes);
    return SemId;
    }

/********************************************************************
 * RTDS_SemCCreate
 *-------------------------------------------------------------------
 * This function creates a counting semaphore
 *
 * PARAMETERS
 *     initialState : int    Initial semaphore counter value 
 *
 * RETURN
 *     RTDS_SemType          Pointer on the created semaphore
 *********************************************************************/
RTDS_SemType *RTDS_SemCCreate( int initialState )
    {
    RTDS_SemType *SemId = NULL;
    
    SemId = ( RTDS_SemType * )RTDS_MALLOC( sizeof( RTDS_SemType ) );
    SemId->typeName = COUNTING;

    if ( sem_init( &( SemId->semId.stdlib_SemType ) , POSIX_SEMAPHORE_MONOPROCESS , initialState ) != 0 )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_INIT )
        }

    return SemId;
    }

/********************************************************************
 * RTDS_SemBCreate
 *-------------------------------------------------------------------
 * This function creates a binary semaphore
 *
 * PARAMETERS:
 *     initialState : int initial state when creating 0: Empty / 1: Full
 *
 * RETURN:
 *     Pointer on the created semaphore: SemaphoreId
 *********************************************************************/
RTDS_SemType *RTDS_SemBCreate( int initialState )
    {
    RTDS_SemType *SemId = NULL;
    
    SemId = ( RTDS_SemType * )RTDS_MALLOC( sizeof( RTDS_SemType ) );
    SemId->typeName = BINARY;

    if( sem_init( &( SemId->semId.stdlib_SemType ) , POSIX_SEMAPHORE_MONOPROCESS , initialState ) != RTDS_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_INIT )
        }
    return SemId;
    }

/********************************************************************
 * RTDS_SemDelete
 *-------------------------------------------------------------------
 * This function deletes a previously created semaphore
 *
 * PARAMETERS:
 *     semId : RTDS_SemType    Semaphore Id
 *********************************************************************/
void RTDS_SemDelete( RTDS_SemType *semId )
    {
    int rvalue = -1;
    
    switch( semId->typeName )
        {
        case COUNTING : 
        case BINARY :
            rvalue = sem_destroy( &( semId->semId.stdlib_SemType ) );
            break;
            
        case MUTEX : 
            rvalue = pthread_mutex_destroy( &( semId->semId.posixlib_SemType ) );
            break;
            
        default :
            break;
         }   
    if( rvalue != 0 )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_DELETE )
        }
    RTDS_FREE( semId );
    }

/********************************************************************
 * RTDS_SemGive
 *-------------------------------------------------------------------
 * This function releases a semaphore
 *
 * PARAMETERS:
 *     semId : RTDS_SemType    Semaphore Id
 *
 * RETURN:
 *     RTDS_OK or RTDS_ERROR
 *********************************************************************/
int RTDS_SemGive( RTDS_SemType *semId )
    {
    int rvalue;
    
    switch(semId->typeName)
        {
        case COUNTING :
        case BINARY :
            rvalue = sem_post( &( semId->semId.stdlib_SemType ) );
            break;
        
        case MUTEX : 
            rvalue = pthread_mutex_unlock( &( semId->semId.posixlib_SemType ) );
            break;
        
        default :
            break;
        }
    if (rvalue == 0)
			return RTDS_OK;
    else
			return RTDS_ERROR;
    }

/********************************************************************
 * RTDS_SemTake
 *-------------------------------------------------------------------
 * This function takes a semaphore
 *
 * PARAMETERS:
 *   semId : RTDS_SemType  Semaphore Id
 *   time_to_Wait : long   0 (no wait) 
 *                         or RTDS_SEMAPHORE_TIME_OUT_FOREVER 
 *                         otherwise RTDS_SYSTEM_ERROR
 *
 * RETURN:
 *   Function status RTDS_OK if take successful or RTDS_TIMEOUT if
 *   semaphore timed out
 *********************************************************************/
int RTDS_SemTake( RTDS_SemType  *semId , long time_to_Wait )
    {
    int rvalue = RTDS_ERROR;

    if ( (semId->typeName == COUNTING) || (semId->typeName == BINARY) )
        {
        /* Make a loop because any posix signal might interrupt the wait... */
        do
            {
            if ( time_to_Wait == 0 )
                {
                rvalue = sem_trywait( &( semId->semId.stdlib_SemType ) );
                }
            else if ( time_to_Wait == RTDS_SEMAPHORE_TIME_OUT_FOREVER )
                {
                rvalue = sem_wait( &( semId->semId.stdlib_SemType ) );
                }
            else
            				{
	            	 RTDS_SYSTEM_ERROR ( RTDS_ERROR_SEM_TAKE_TIMEOUT_VALUE );
	            	 }
            } while ( ( rvalue != 0 ) && ( errno == EINTR ) ); 
        }
    else 
        {
        if ( semId->typeName == MUTEX )
            {
            if (time_to_Wait == 0)
                {
                rvalue = pthread_mutex_trylock( &( semId->semId.posixlib_SemType ) );
                }
            else if ( time_to_Wait == RTDS_SEMAPHORE_TIME_OUT_FOREVER )
                {
                rvalue = pthread_mutex_lock( &( semId->semId.posixlib_SemType ) );
                }
            else
	            		{
	            	 RTDS_SYSTEM_ERROR ( RTDS_ERROR_SEM_TAKE_TIMEOUT_VALUE );
	            	 }
            }
        }
    if ( rvalue == 0 )
      return RTDS_OK;
    else
    		{
    	 if ( ( errno == EAGAIN ) || (errno == 0) || (rvalue == EBUSY) )
    	 		return RTDS_TIMEOUT;
    	 else
    	 		RTDS_SYSTEM_ERROR ( RTDS_ERROR_SEM_TAKE );
    	 }
    }

/********************************************************************
 * RTDS_MsgQueueCreate
 *-------------------------------------------------------------------
 * This function is to allocate memory for both semaphore:
 * Chained_List_Sem (that protects Chained list) and Blocking_Sem
 * (that allows thread to wait for messages when message box is empty)
 *
 * This structure contains pointers to semaphore and pointer to message
 * box (chained list)
 *
 * RETURN:
 *     RTDS_QCB* 
 *********************************************************************/
RTDS_QCB * RTDS_MsgQueueCreate()
    {
    RTDS_QCB *Queue_Control_Block;

    /* Initialization of the RTDS_QCB (Queue Control Block) */
    Queue_Control_Block=(RTDS_QCB*)RTDS_MALLOC(sizeof(RTDS_QCB));

    /* Initialization of the Chained_listSem */
    /* Create semaphore only for this process and with initial value: unlocked */
    Queue_Control_Block->chainedListSemId = RTDS_SemMCreate();

    /* Initialization of the Blocking_Sem */
    Queue_Control_Block->blockingSemId = RTDS_SemCCreate( POSIX_SEMAPHORE_LOCKED );

    /* Initialize Queue_Control_Block->queue to NULL */
    Queue_Control_Block->queue = NULL;

    return Queue_Control_Block;
    }

/********************************************************************
 * RTDS_MsgQ_Send
 *-------------------------------------------------------------------
 * Allocate memory for new message to send
 * Change the "next" pointer of the last message to point to the new
 * message
 *
 * PARAMETERS:
 *     receiver :           RTDS_QCB*           Pointer to the message queue of the receiver
 *     RTDS_messageToSend : RTDS_MessageHeader* Message to send
 *
 * RETURN:
 *     Function status 0 if succedded -1 otherwise
 ********************************************************************/
int RTDS_MsgQSend(
	RTDS_QCB            *receiver ,
	RTDS_MessageHeader  *RTDS_messageToSend )
    {
    RTDS_MessageHeader *tmpMsg = NULL;
    RTDS_MessageHeader *dynamicMessageToSend;

    if ( RTDS_messageToSend != NULL )
        {
        dynamicMessageToSend = ( RTDS_MessageHeader * )RTDS_MALLOC( sizeof( RTDS_MessageHeader ) );
        memcpy( dynamicMessageToSend , RTDS_messageToSend , ( size_t )sizeof( RTDS_MessageHeader ) );

        /* Protect Chained list */
				RTDS_SemTake( ( RTDS_SemType * )receiver->chainedListSemId , RTDS_SEMAPHORE_TIME_OUT_FOREVER );

        /* initialize tmpMsg to Point on the first message */
        tmpMsg = receiver->queue;

        if( tmpMsg != NULL )
            {
            while( tmpMsg->next != NULL )
                {
                /* Set tmpMsg to point on the last message */
                tmpMsg = tmpMsg->next;
                }
            /* Set the last message's "next" pointer to the adress of the message to send (dynamicMessageToSend) */
            tmpMsg->next = dynamicMessageToSend;
            }
        else
            {
            receiver->queue = dynamicMessageToSend;
            }

        /* Give Chained list semaphore */
        if( RTDS_SemGive( ( RTDS_SemType * )receiver->chainedListSemId ) == RTDS_ERROR )
            {
            return RTDS_ERROR;
            }

        /* Give Blocking semaphore to allow thread to read message */
        if ( RTDS_SemGive( ( RTDS_SemType * )receiver->blockingSemId ) == RTDS_ERROR )
            {
            return RTDS_ERROR;
            }
        }
    return RTDS_OK;
    }

/********************************************************************
 * RTDS_MsgQRead
 *-------------------------------------------------------------------
 * Read a message queue and returns a pointer and the size of the
 * buffer containing the received data.
 *
 * PARAMETERS:
 *     queueId : RTDS_QCB*            Pointer to the message queue of the receiver
 *     message : char*                Pointer to copy message readed
 *
 * RETURN:
 *     Function status 0 if succedded -1 otherwise
 ********************************************************************/
int RTDS_MsgQRead(
 RTDS_QCB *queueId ,
 char *message )
    {
    RTDS_MessageHeader *msg;
    #ifdef RTDS_SIMULATOR
      if ( queueId->queue == NULL )
        {
        /* No more messages. Stops if run until rtos message queue empty. */
        RTDS_DummyRunUntilQueueEmptyFunction();
        }
    #endif
    /* Wait for a message to read */
    RTDS_SemTake( ( RTDS_SemType * )queueId->blockingSemId , RTDS_SEMAPHORE_TIME_OUT_FOREVER );
    /* Protect Chained list */
    RTDS_SemTake( ( RTDS_SemType * )queueId->chainedListSemId , RTDS_SEMAPHORE_TIME_OUT_FOREVER );
    /* Copy message to destination and free memory */
    if ( queueId->queue != NULL )
        {
        msg = queueId->queue;
        memcpy(message,msg,sizeof(RTDS_MessageHeader));
        queueId->queue = msg->next;
        RTDS_FREE(msg);
        }
    else
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSGQ_READ );
        }
    /* Give Chained list semaphore*/
    if( RTDS_SemGive( ( RTDS_SemType * )queueId->chainedListSemId ) == RTDS_ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSGQ_READ )
        }
    return RTDS_OK;
    }

/********************************************************************
 * RTDS_MsgQDelete
 *-------------------------------------------------------------------
 * This function deletes a message queue:
 *     - delete all messages contained in the queue
 *     - delete the queue (and related semaphore)
 *
 * PARAMETERS:
 *     queueId : RTDS_QCB*    Message queue to delete
 *********************************************************************/
void RTDS_MsgQDelete( RTDS_QCB *queueId )
    {
    RTDS_MessageHeader *msg, *tmpMsg;

    /* 1- DELETE ALL MESSAGES */
    msg=queueId->queue;

    /* Delete all messages contained in the queue */
    while ( msg != NULL )
        {
        tmpMsg = msg->next;
        RTDS_FREE( msg );
        msg = tmpMsg;
        }

    /* 2- DELETE QUEUE CONTROL BLOCK */
    /* Free memory */
    RTDS_SemDelete( ( RTDS_SemType * )queueId->chainedListSemId );
    RTDS_SemDelete( ( RTDS_SemType * )queueId->blockingSemId );
    RTDS_FREE( queueId );
    }

/********************************************************************
 * RTDS_TaskCreate
 *-------------------------------------------------------------------
 * Creates a new thread.
 * If any error is encountered RTDS_SYSTEM_ERROR is called and
 * should stop execution.
 *
 * PARAMETERS:
 *     priority : int             Message queue to delete
 *     functionAddress : void*    Adresse of function to execute
 *     newProcessInfo : void*     Parameters for the created thread
 *
 * RETURN:
 *     RTDS_RtosTaskId            ID of the created task 
 *********************************************************************/
RTDS_RtosTaskId RTDS_TaskCreate(
 int priority ,
 void *( *functionAddress )( void* ) ,
 void *newProcessInfo )
    {
    /* Attributes for threads */
    pthread_attr_t attr1;
    /* The thread themselves */
    RTDS_RtosTaskId th1;   

    /* Test if priority parameter is valid argument */
    if( ( priority < sched_get_priority_min( SCHED_RR ) ) || ( priority > sched_get_priority_max( SCHED_RR ) ) )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_WRONG_PRIORITY_VALUE )
        }

    if( pthread_attr_init( &attr1 ) != RTDS_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PTHREAD_ATTR_INIT )
        }
    if( pthread_attr_setdetachstate( &attr1 , PTHREAD_CREATE_DETACHED ) != RTDS_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PTHREAD_ATTR_SETDETACH )
        }

    #ifdef RTDS_THREAD_PRIO_ENABLE
        /*    The realtime scheduling policies SCHED_RR and SCHED_FIFO are available only
         *    with superuser privileges. This macro allows user to create thread this priority disable
         */
        {
        struct sched_param param1;
        if( pthread_attr_setschedpolicy( &attr1 , SCHED_RR ) != RTDS_OK )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_PTHREAD_SETSCHEDPOLICY )
            }
        param1.sched_priority = priority;

        if( pthread_attr_setschedparam( &attr1 , &param1 ) != RTDS_OK )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_PTHREAD_ATTR_SETSCHEDPARAM )
            }
        }
    #endif

    /* CREATE & RUN THREAD */
    if( pthread_create( &th1 , &attr1 ,( void *( * )( void * ) )functionAddress , newProcessInfo ) != RTDS_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PTHREAD_CREATE )
        }

    if( pthread_attr_destroy( &attr1 ) != RTDS_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PTHREAD_ATTR_DESTROY )
        }
    return th1;
    }

/********************************************************************
 * RTDS_TimerDelete()
 *-------------------------------------------------------------------
 * This function stop a timer thread if possible
 *
 * PARAMETERS:
 *     RTDS_TimerState : watchDogId    Timer to delete
 *
 * RETURN:
 *     RTDS_OK if succedded RTDS_ERROR otherwise
 ********************************************************************/
int RTDS_TimerDelete( RTDS_TimerState *timerInfo )
    {
    /* Disable timer thread deletion when RTDS_DISCRETE_TIME is not set */
    #ifndef RTDS_DISCRETE_TIME
        if( timerInfo->state == RTDS_TIMER_OK )
            {
            /* Try to cancel the thread */
            if ( pthread_cancel( (pthread_t) (timerInfo->watchDogId) ) == 0 )
                return RTDS_OK;
            else
                return RTDS_ERROR;
            }
        else
            {
            return RTDS_ERROR;
            }
    #else
        return RTDS_OK;
    #endif
    }

/********************************************************************
 * RTDS_GetCurrentThread
 *-------------------------------------------------------------------
 * Returns the current thread id
 *
 * RETURN
 *     RTDS_RtosTaskId    threadId
 ********************************************************************/
RTDS_RtosTaskId RTDS_GetCurrentThread()
    {
    return ( RTDS_RtosTaskId )pthread_self();
    }

/********************************************************************
 * RTDS_TickGet
 *-------------------------------------------------------------------
 * Get the system time (global variable RTDS_SystemTime)
 * 
 * RETURN:
 *     unsigned long    System time
 ********************************************************************/
unsigned long RTDS_TickGet()
    {
    #ifndef RTDS_DISCRETE_TIME
        struct timeval timeValue;
        struct timezone not_used;
        unsigned long varSec;
        unsigned long varuSec;
        unsigned long newTime;
        if( gettimeofday( &timeValue , &not_used ) != RTDS_OK )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_GETTIME )
            }
        varSec = ( unsigned long )timeValue.tv_sec;
        varuSec = ( unsigned long )timeValue.tv_usec;

        newTime = timeValue.tv_sec * 1000 + timeValue.tv_usec / 1000 - RTDS_globalStartTime;
        RTDS_globalSystemTime = newTime;
        return newTime;
    #else
        return RTDS_globalSystemTime;
    #endif
    }

/********************************************************************
 * RTDS_TickSet
 *-------------------------------------------------------------------
 * This function set the system time
 * therefore this function set the extern variable (RTDS_globalSystemTime)
 * defined in the main function using System Time MACRO (RTDS_MACRO.h)
 *
 * PARAMETERS:
 *     timeValue : unsigned long    Time to set  
 ********************************************************************/
void RTDS_TickSet( unsigned long timeValue )
    {
    #ifndef RTDS_DISCRETE_TIME
        RTDS_globalStartTime = RTDS_TickGet() - timeValue;
    #endif
    RTDS_globalSystemTime = timeValue;
    }
   
/********************************************************************
 * RTDS_GetTimerUniqueId
 *-------------------------------------------------------------------
 * Get an available timer unique id. Is used to set timers
 *
 * PARAMETERS:
 *     timerList : RTDS_TimerState*    Points to the first element of 
 *                                     the chained list of RTDS_TimerState
 * RETURN:
 *     long   Timer unique ID 
 * **************************************************************** *
 * Context:
 *     The chained list of RTDS_TimerState is sorted by increasing
 *     unique timer id
 *     timerUniqueId = NULL means the message is not a timer; so it
 *     is avoided.
 * **************************************************************** */
long RTDS_GetTimerUniqueId( RTDS_TimerState *timerList )
    {
    RTDS_TimerState *RTDS_prevTimer;
    RTDS_TimerState *RTDS_timer;
    long newTimerId;

    RTDS_CRITICAL_SECTION_START;
    /* If list is empty take 1 */
    if ( timerList == NULL )
        {
        newTimerId = 1;
        }
    /* If 1 is available, take it */
    else 
        {
        if ( timerList->timerUniqueId != 1 )
            {
            newTimerId = 1;
            }
        else
            {
            /* If there is a gap in the timerId chained list, take an Id in the gap */
            RTDS_prevTimer = timerList;
            newTimerId = 0;
            for ( RTDS_timer = timerList->next ; RTDS_timer != NULL ; RTDS_timer = RTDS_timer->next )
                {
                if ( RTDS_timer->timerUniqueId != RTDS_prevTimer->timerUniqueId + 1 )
                    {
                    newTimerId = RTDS_prevTimer->timerUniqueId + 1;
                    }
                RTDS_prevTimer = RTDS_timer;
                }
            /* No gap, let's take the next value */
            if ( newTimerId == 0 )
                {
                newTimerId = RTDS_prevTimer->timerUniqueId + 1;
                }
            }
        }    
    RTDS_CRITICAL_SECTION_STOP;
    /* Check the counter did not go back to 0 */
    if ( newTimerId == 0 )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_NO_MORE_TIMER_UNIQUE_ID )
        }
    return newTimerId;
    }

/********************************************************************
 * RTDS_StartTimer
 *-------------------------------------------------------------------
 * Starts a watchdog with the necessary parameters to create the
 * timer when it goes off.
 *
 * PARAMETERS:
 *     instanceId : RTDS_SdlInstanceId          Instance id for the receiver
 *     timerNumber : long                       TimerNumber
 *     timerUniqueId : long                     TimerUniqueId
 *     delay : int                              Delay of the timer
 *     pTimerStateList : RTDS_TimerState**      TimerStateList: Address of the list of timer
 * **************************************************************** */
void RTDS_StartTimer(
 RTDS_SdlInstanceId *instanceId ,
 long timerNumber ,
 long timerUniqueId , 
 int delay ,
 RTDS_TimerState **pTimerStateList ,
 RTDS_GlobalProcessInfo *RTDS_currentContext )
    {
    RTDS_TimerState *timerState;
    RTDS_TimerState *previousTimerState;
    RTDS_TimerState *newTimerState;
    
    /* Update the list of timers */
    /* Important note: it is a sorted list based on the timerUniqueId field */
    newTimerState = ( RTDS_TimerState * )RTDS_MALLOC( sizeof( RTDS_TimerState ) );

    /* Initialize the new element */
    newTimerState->state = RTDS_TIMER_OK;
    newTimerState->timerNumber = timerNumber;
    newTimerState->timeoutValue = delay + RTDS_TickGet();
    newTimerState->timerUniqueId = timerUniqueId;
    newTimerState->next = NULL;
    newTimerState->receiverId = instanceId;

    /* Insert the new element */
    if ( *pTimerStateList == NULL ) /* The list is empty */
        {
        *pTimerStateList = newTimerState;
        }
    else    /* The list is not empty */
        {
        previousTimerState = NULL;
        for ( timerState = *pTimerStateList ; timerState != NULL ; timerState = timerState->next )
            {
            if ( timerState->timerUniqueId > timerUniqueId )
                {
                if ( previousTimerState == NULL )
                    {
                    *pTimerStateList = newTimerState;
                    newTimerState->next = timerState;
                    }
                else
                    {
                    previousTimerState->next = newTimerState;
                    newTimerState->next = timerState;
                    }
                break;
                }
            previousTimerState = timerState;
            }

        if ( timerState == NULL ) /* Inserted at the end of the list */
            {
            previousTimerState->next = newTimerState;
            }
        }

    RTDS_SIMULATOR_TRACE( RTDS_timerStarted, newTimerState, delay, RTDS_currentContext );
    /* CREATE NEW TIMER THREAD IF RTDS_DISCRETE_TIME IS NOT SET */
    #ifndef RTDS_DISCRETE_TIME
        newTimerState->watchDogId = ( RTDS_RtosTaskId )RTDS_TaskCreate( RTDS_TIMER_PRIORITY , ( void *( * )( void * ) )RTDS_TimerFunction , ( RTDS_TimerState * )newTimerState );
    #else
        /* if time is discrete: set Timer threadId to RTDS_ERROR */
        newTimerState->watchDogId = ( RTDS_RtosTaskId )RTDS_ERROR;
    #endif
    }




/********************************************************************
 * RTDS_WatchDogFunction
 *-------------------------------------------------------------------
 * Send a timer message using the parameters given to the watchdog
 *
 * PARAMETERS:
 *     parameter: void*    Needed parameters to create and send the timer:
 *                         - QueueId of the receiver
 *                         - TimerNumber
 *                         - TimerUniqueId
 *                         - WatchDogId to destroy it
 * **************************************************************** */
void RTDS_WatchDogFunction( RTDS_TimerState * ptimerState )
    {
    RTDS_TimerState * parameter;
    RTDS_MessageHeader *newMessage;

    parameter = ( RTDS_TimerState * )ptimerState;
    newMessage = ( RTDS_MessageHeader * )RTDS_MALLOC( sizeof( RTDS_MessageHeader ) );

    /* Fill in the message parameters */
    #if defined( RTDS_SIMULATOR )
        newMessage->messageUniqueId = 0;
    #endif
    newMessage->messageNumber = parameter->timerNumber;
    newMessage->timerUniqueId = parameter->timerUniqueId;
    newMessage->sender = parameter->receiverId;
    newMessage->receiver = parameter->receiverId;
    newMessage->next = NULL;
    newMessage->dataLength = 0;
    newMessage->pData = NULL;

    /* Ignore if process has vanished */
    RTDS_CRITICAL_SECTION_START;
    if( RTDS_CheckQueueId( parameter->receiverId ) != RTDS_ERROR )
        {
        /* Send the message */
        if ( RTDS_MsgQSend( parameter->receiverId->queueId, newMessage ) == RTDS_ERROR )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_Q_SEND_IN_WATCHDOG )
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    RTDS_FREE( newMessage );
    }



/********************************************************************
 * RTDS_TimerFunction
 *-------------------------------------------------------------------
 * This function waits for a delay and then call
 * a function : All timer information are transmitted with the
 * RTDS_TimerState structure (function parameter)
 *
 * PARAMETERS:
 *     timerInfo : RTDS_TimerState*    Timer information
 ********************************************************************/
void RTDS_TimerFunction( RTDS_TimerState *timerInfo )
    {
    struct timespec ts;
    struct timespec remainingTime;
    float exact_delay;
    int second;
    int nsecond;

    exact_delay = timerInfo->timeoutValue-RTDS_TickGet();
    second = ( int )( exact_delay / 1000 );
    exact_delay = exact_delay - second * 1000;
    nsecond =( int ) exact_delay * 1000000;

    ts.tv_sec = second;
    ts.tv_nsec = nsecond;

    /* Repeat until nanosleep returns 0 */
    /* This is done because nanosleep catches
         all signals (debugger, handler...) */
    while ( nanosleep( &ts , &remainingTime ) != 0 )
        {
        ts.tv_sec = remainingTime.tv_sec;
        ts.tv_nsec = remainingTime.tv_nsec;
        }

    RTDS_WatchDogFunction( timerInfo );

    }

/********************************************************************
 * RTDS_StopTimer
 *-------------------------------------------------------------------
 * Stops a timer in trying to delete the watchdog. If unsuccessfull
 * set it cancelled in the timer chained list verified by the RTDS
 * kernel
 *
 * PARAMETERS:
 *     timerNumber : long                              Timer identifier
 *     pTimerStateList : RTDS_TimerState**             Pointer to the timer chained list
 *     RTDS_currentContext : RTDS_GlobalProcessInfo*   Pointer to the current context of the task   
 * **************************************************************** */
void RTDS_StopTimer( long timerNumber , RTDS_TimerState **pTimerStateList , RTDS_GlobalProcessInfo *RTDS_currentContext )
    {
    RTDS_TimerState *RTDS_timer;
    RTDS_TimerState *RTDS_prevTimer;
    int DeleteStatus;
    RTDS_MessageHeader *RTDS_message;
    RTDS_MessageHeader *RTDS_prevMessage;

    RTDS_prevTimer = NULL;
    for ( RTDS_timer = *pTimerStateList ; RTDS_timer != NULL ; RTDS_timer = RTDS_timer->next )
        {
        if ( ( RTDS_timer->timerNumber == timerNumber ) && ( RTDS_timer->state != RTDS_TIMER_CANCELLED ) )
            {
            RTDS_SIMULATOR_TRACE( RTDS_timerCancelled , RTDS_timer , 0 , RTDS_currentContext );

            #ifndef RTDS_POSIX_TIMER_THREAD_PROBLEM
                /* Let's try to cancel the watchdog */
                DeleteStatus = RTDS_TimerDelete( RTDS_timer );
                if ( DeleteStatus == RTDS_OK )
                    {
                    /* Remove it from the list */
                    /* Is the first of the list */
                    if (RTDS_prevTimer == NULL)
                        {
                        *pTimerStateList = RTDS_timer->next;
                        }
                    else    /* Not the first of the list */
                        {
                        RTDS_prevTimer->next = RTDS_timer->next;
                        }
                    RTDS_FREE( RTDS_timer );
                    }
                    /* Could not cancel the timer. Probably went off allready */
                    /* Set it cancelled in the list */
                else
                    {
                    RTDS_timer->state = RTDS_TIMER_CANCELLED;
                    }
            #else
                RTDS_timer->state = RTDS_TIMER_CANCELLED;
            #endif
            return;
            }
        RTDS_prevTimer = RTDS_timer;
        }

        /* If execution gets here: the timer might be in the save queue */
        RTDS_prevMessage = NULL;
        for ( RTDS_message = RTDS_currentContext->readSaveQueue ; RTDS_message != NULL ; RTDS_message = RTDS_message->next )
            {
            if ( RTDS_message->messageNumber == timerNumber )
                {
                /* Remove it from the list */
                /* Is the first of the list */
                if ( RTDS_prevMessage == NULL )
                    {
                    RTDS_currentContext->readSaveQueue = RTDS_message->next;
                    }
                /* Not the first of the list */
                else
                    {
                    RTDS_prevMessage->next = RTDS_message->next;
                    }
                RTDS_FREE( RTDS_message );
                return;
                }
            RTDS_prevMessage = RTDS_message;
            }

        /* If execution gets here: the timer might be in the save queue */
        RTDS_prevMessage = NULL;
        for ( RTDS_message = RTDS_currentContext->writeSaveQueue ; RTDS_message != NULL ; RTDS_message = RTDS_message->next )
            {
            if( RTDS_message->messageNumber == timerNumber )
                {
                /* Remove it from the list */
                /* Is the first of the list */
                if( RTDS_prevMessage == NULL )
                    {
                    RTDS_currentContext->writeSaveQueue = RTDS_message->next;
                    }
                /* Not the first of the list */
                else
                    {
                    RTDS_prevMessage->next = RTDS_message->next;
                    }
                RTDS_FREE( RTDS_message );
                return;
                }
            RTDS_prevMessage = RTDS_message;
        }
    }

/********************************************************************
 * RTDS_GetProcessQueueId
 *-------------------------------------------------------------------
 * Returns the queueId of process based on its name
 *
 * PARAMETERS:
 *     processNumber : int    Process name as a number defined in RTDS_gen.h*
 *
 * RETURN:
 *     RTDS_SdlInstanceId    Process instance id if found , NULL otherwise
 * **************************************************************** */
RTDS_SdlInstanceId *RTDS_GetProcessQueueId(int sdlProcessNumber)
    {
    RTDS_GlobalProcessInfo *processInfo;
    RTDS_SdlInstanceId *foundQueue = NULL;

    RTDS_CRITICAL_SECTION_START;
    for ( processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next )
        {
        if ( processInfo->sdlProcessNumber == sdlProcessNumber )
            {
            foundQueue = processInfo->mySdlInstanceId;
            break;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;

    if ( foundQueue == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_PROCESS_QUEUE_ID )
        }
    return foundQueue;
    }

/********************************************************************
 * RTDS_CheckQueueId
 *-------------------------------------------------------------------
 * Returns RTDS_OK is queueId exist or ERROR if not
 *
 * PARAMETERS:
 *     instanceId : RTDS_SdlInstanceId    Process instance id
 *
 * RETURN:
 *     int    RTDS_OK if exist RTDS_ERROR otherwise
 * **************************************************************** */
int RTDS_CheckQueueId( RTDS_SdlInstanceId * instanceId )
    {
    RTDS_GlobalProcessInfo *processInfo;

    for ( processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next )
        {
        if ( processInfo->mySdlInstanceId == instanceId )
            {
            return RTDS_OK;
            }
        }
    return RTDS_ERROR;
    }

/* **************************************************************** *
 * RTDS_MsgQueueSend
 *-------------------------------------------------------------------
 * Send a message in a process's queue
 *
 * PARAMETERS:
 *     messageNumber : long                             MessageNumber representing a message name
 *     dataLength : long                                DataLength length of data pointed by pData
 *     pData : unsigned char *                          Pointer on data sent with the message
 *     receiver : RTDS_SdlInstanceId *                  Message receiver instance id
 *     sender : RTDS_InstanceId *                       Message sender instance id
 *     RTDS_currentContext : RTDS_GlobalProcessInfo *   Current context of the task
 * **************************************************************** */
void RTDS_MsgQueueSend(
    long messageNumber ,
    long dataLength ,
    unsigned char *pData ,
    RTDS_SdlInstanceId *receiver ,
    RTDS_SdlInstanceId *sender ,
    RTDS_GlobalProcessInfo *RTDS_currentContext )
    {
    RTDS_MessageHeader RTDS_messageToSend;

    RTDS_messageToSend.messageNumber = messageNumber;
    RTDS_messageToSend.timerUniqueId = 0;
    RTDS_messageToSend.sender = sender;
    RTDS_messageToSend.receiver = receiver;
    RTDS_messageToSend.dataLength = dataLength;
    RTDS_messageToSend.pData = pData;
    RTDS_messageToSend.next = NULL;
    #if defined( RTDS_SIMULATOR )
        RTDS_messageToSend.messageUniqueId = RTDS_GetMessageUniqueId();
    #endif

    RTDS_SIMULATOR_TRACE( RTDS_messageSent , &RTDS_messageToSend , receiver , RTDS_currentContext );

    RTDS_CRITICAL_SECTION_START;
    /* Check if process still exist */
    if( RTDS_CheckQueueId( receiver ) == RTDS_OK )
        {
        if ( RTDS_MsgQSend( receiver->queueId , &RTDS_messageToSend ) == RTDS_ERROR )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_SEND )
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    }

/* **************************************************************** *
 * RTDS_TTCNMsgQueueSend
 *-------------------------------------------------------------------
 * Send a message in a TTCN port
 *
 * PARAMETERS:
 *     messageNumber : long                             MessageNumber representing a message name
 *     dataLength : long                                DataLength length of data pointed by pData
 *     pData : unsigned char *                          Pointer on data sent with the message
 *     portId : RTDS_RtosQueueId                        Message receiver queue id
 *     receiver : RTDS_SdlInstanceId *                  Message receiver instance id
 *     sender : RTDS_InstanceId *                       Message sender instance id
 *     RTDS_currentContext : RTDS_GlobalProcessInfo *   Current context of the task
 * **************************************************************** */
void RTDS_TTCNMsgQueueSend(
    long messageNumber ,
    long dataLength ,
    unsigned char *pData ,
    RTDS_RtosQueueId portId ,
    RTDS_SdlInstanceId *receiver ,
    RTDS_SdlInstanceId *sender ,
    RTDS_GlobalProcessInfo *RTDS_currentContext )
    {
    RTDS_MessageHeader RTDS_messageToSend;

    RTDS_messageToSend.messageNumber = messageNumber;
    RTDS_messageToSend.timerUniqueId = 0;
    RTDS_messageToSend.sender = sender;
    RTDS_messageToSend.receiver = receiver;
    RTDS_messageToSend.dataLength = dataLength;
    RTDS_messageToSend.pData = pData;
    RTDS_messageToSend.next = NULL;
    #if defined( RTDS_SIMULATOR )
        RTDS_messageToSend.messageUniqueId = RTDS_GetMessageUniqueId();
    #endif

    RTDS_SIMULATOR_TRACE( RTDS_messageSent , &RTDS_messageToSend , receiver , RTDS_currentContext );

    RTDS_CRITICAL_SECTION_START;

    if ( RTDS_MsgQSend( portId , &RTDS_messageToSend ) == RTDS_ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_SEND )
        }
    RTDS_CRITICAL_SECTION_STOP;
    }
    
    
/* **************************************************************** *
 * RTDS_MsgQueueReceive
 *-------------------------------------------------------------------
 * Receives a message from a process's queue
 *
 * PARAMETERS:
 *     instanceId : RTDS_SdlInstanceId*  Instance id to read the messsage from
 *     message : char*            Pointer on the message to receive
 *     size : int                 Size of message
 *     mode : int                 Mode used to read (blocking or non blocking)
 * **************************************************************** */
void RTDS_MsgQueueReceive(RTDS_SdlInstanceId * instanceId, char *message, int size, int mode)
    {
    if ( RTDS_MsgQRead( instanceId->queueId , message ) == RTDS_ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_RECEIVE )
        }
    }

/* **************************************************************** *
 * RTDS_TTCNMsgQueueReceive
 *-------------------------------------------------------------------
 * Receives a message from a TTCN queue
 *
 * PARAMETERS:
 *     portId : RTDS_RtosQueueId  Queue id to read the message from
 *     instanceId : RTDS_SdlInstanceId*  Instance id to read the messsage from   
 *     message : char*            Pointer on the message to receive
 *     size : int                 Size of message
 *     mode : int                 Mode used to read (blocking or non blocking)
 * **************************************************************** */
void RTDS_TTCNMsgQueueReceive(RTDS_RtosQueueId portId, RTDS_SdlInstanceId * instanceId, char *message, int size, int mode)
    {
    if ( RTDS_MsgQRead( portId , message ) == RTDS_ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_RECEIVE )
        }
    }
    
/* **************************************************************** *
 * RTDS_ProcessCreate
 *-------------------------------------------------------------------
 * Create a new SDL process and add a processInfo struct in the
 * process info chained list
 *
 * PARAMETERS:
 *     processNumber : int                            Name of the process as a number
 *     functionAddress : void*                        Address of the process function
 *     priority : int                                 Priority fo the process
 *     pOffspringId : RTDS_SdlInstanceId**               Address of the instance id to be created
 *     selfId : RTDS_SdlInstanceId*                      Instance id for the parent process
 *     synchronization : short
 *     RTDS_currentContext : RTDS_GlobalProcessInfo*  Pointer to the current context of the task
 *
 * RETURNS:
 *     Nothing but updates the OFFSPRING of the caller
 * **************************************************************** */
void RTDS_ProcessCreate(
  int                       processNumber,
  void                    * functionAddress,
  int                       priority,
  RTDS_SdlInstanceId         **pOffspringId,
  RTDS_SdlInstanceId         * selfId,
  short                     synchronization,
  RTDS_GlobalProcessInfo  * RTDS_currentContext
)
    {
    RTDS_GlobalProcessInfo *processInfo;
    RTDS_GlobalProcessInfo *newProcessInfo;

    /* Allocate and fill in a new processInfo structure */
    newProcessInfo = ( RTDS_GlobalProcessInfo * )RTDS_MALLOC( sizeof( RTDS_GlobalProcessInfo ) );

    /* Sets the process name as a number */
    newProcessInfo->sdlProcessNumber = processNumber;
    /* SDL initial state */
    newProcessInfo->sdlState = 0;
    /* Set SDL task priority */
#ifndef RTDS_THREAD_PRIO_ENABLE
    newProcessInfo->priority = -1;
#else
    newProcessInfo->priority = priority;
#endif

    /* Last structure of the chained list */
    newProcessInfo->next = NULL;
    newProcessInfo->parentSdlInstanceId = selfId;
    newProcessInfo->offspringSdlInstanceId = NULL;
    newProcessInfo->currentMessage = NULL;
    newProcessInfo->timerList = NULL;
    newProcessInfo->readSaveQueue = NULL;
    newProcessInfo->writeSaveQueue = NULL;

    /* New task's queue creation (POSIX)*/
    *pOffspringId = (RTDS_SdlInstanceId*)RTDS_MALLOC(sizeof(RTDS_SdlInstanceId));
    (*pOffspringId)->queueId = RTDS_MsgQueueCreate();
    (*pOffspringId)->instanceNumber = 0;
    newProcessInfo->mySdlInstanceId = *pOffspringId;

    /* Task creation */
     newProcessInfo->myRtosTaskId = RTDS_TaskCreate( priority ,( void *( * )( void * ) )functionAddress , ( void* )newProcessInfo );

    /* Add the process information to the chained list pointed by the RTDS_globalProcessInfo global variable */
    RTDS_CRITICAL_SECTION_START;
    if ( RTDS_globalProcessInfo == NULL )
        {
        RTDS_globalProcessInfo = newProcessInfo;
        }
    else
        {
        /* Let's get to the end of the list */
        processInfo = RTDS_globalProcessInfo;
        while( processInfo->next != NULL )
            {
            processInfo = processInfo->next;
            }
        processInfo->next = newProcessInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    RTDS_SIMULATOR_TRACE( RTDS_processCreated , newProcessInfo , 0 , RTDS_currentContext );

    /* The newly created task can now run: RTDS_globalProcessInfo and trace are up to date. */
    if (synchronization == RTDS_HOLD )
        {
        RTDS_START_SYNCHRO_UNHOLD;
        }
    }

/* **************************************************************** *
 * RTDS_ProcessForget
 *-------------------------------------------------------------------
 * Forgets a given process, freeing all associated information
 *
 * PARAMETERS:
 *  - RTDS_currentContext : The current context of the task to delete
 *  - deleteQueue: If true, the message queue for the process is destroyed
 * RETURNS:
 *  - The PID for the forgotten process or 0 if process was not found
 * **************************************************************** */
RTDS_RtosTaskId RTDS_ProcessForget( RTDS_GlobalProcessInfo *RTDS_currentContext, short deleteQueue )
    {
    RTDS_GlobalProcessInfo  * processInfo;
    RTDS_GlobalProcessInfo  * previousProcessInfo;
    RTDS_TimerState         * timer;
    RTDS_MessageHeader      * message;
    RTDS_MessageHeader      * tmpMessage;
    RTDS_SdlInstanceId      * instance2Delete;
    RTDS_RtosTaskId           pidToKill = 0;

    RTDS_CRITICAL_SECTION_START;
    /* Free the current message if any */
    if ( RTDS_currentContext->currentMessage != NULL )
        {
        #if defined( RTDS_SIMULATOR )
            /* Release the message unique id back to the pool */
            RTDS_ReleaseMessageUniqueId( RTDS_currentContext->currentMessage->messageUniqueId );
        #endif
        /* Free the message parameter structure if any */
        if (RTDS_currentContext->currentMessage->pData != NULL){
          RTDS_FREE(RTDS_currentContext->currentMessage->pData);
          }
        /* Free memory */
        RTDS_FREE( RTDS_currentContext->currentMessage );
        }

    /* Clean the timer chained list to free memory and stop watchdogs */
    for ( timer = RTDS_currentContext->timerList ; timer != NULL ; timer = RTDS_currentContext->timerList )
        {
        RTDS_TimerDelete( timer );
        RTDS_currentContext->timerList = timer->next;
        RTDS_FREE( timer );
        }

    /* Clean the save queue: free messages and message unique ids in the read and write save queues */
    message = RTDS_currentContext->readSaveQueue;
    while( message != NULL )
        {
#if defined( RTDS_SIMULATOR )
        RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
#endif
        tmpMessage = message;
        message = message->next;
        RTDS_FREE( tmpMessage );
        }

    message = RTDS_currentContext->writeSaveQueue;
    while( message != NULL )
        {
#if defined( RTDS_SIMULATOR )
        RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
#endif
        tmpMessage = message;
        message = message->next;
        RTDS_FREE( tmpMessage );
        }

    /* Forget and free the process information in the global instance chained list */
    previousProcessInfo = NULL;
    for ( processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next )
        {
        /* If instance was found */
        if ( processInfo->mySdlInstanceId == RTDS_currentContext->mySdlInstanceId )
            {
            RTDS_SIMULATOR_TRACE( RTDS_processDied , processInfo , 0 , RTDS_currentContext );
            /* Remove instance descriptor from list */
            if ( previousProcessInfo == NULL )
                RTDS_globalProcessInfo = processInfo->next;
            else
                previousProcessInfo->next = processInfo->next;
            /* Remember PID for task */
            pidToKill = processInfo->myRtosTaskId;
            /* Delete the message queue if needed */
            instance2Delete = processInfo->mySdlInstanceId;
            if ( deleteQueue )
              RTDS_MsgQDelete( instance2Delete->queueId );
            
            /* Free the process info block */
            RTDS_FREE( instance2Delete );
            RTDS_FREE( processInfo );
            break;
            }
        previousProcessInfo = processInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;

    /* Return found PID */
    return pidToKill;
    }

/* **************************************************************** *
 * RTDS_ProcessKill
 *-------------------------------------------------------------------
 * Kills an SDL process and delete its queue and process info block
 *
 * PARAMETERS:
 *     RTDS_currentContext : RTDS_GlobalProcessInfo*    The current context of the task to delete
 * **************************************************************** */
void RTDS_ProcessKill( RTDS_GlobalProcessInfo *RTDS_currentContext )
    {
    RTDS_RtosTaskId pidToKill;
    
    /* Forget everything about process, delete its message queue and get its PID */
    pidToKill = RTDS_ProcessForget(RTDS_currentContext, 1);
    /* Delete the task itself if possible */
    if ( pidToKill != 0 )
        {
        pthread_exit( NULL );
        }
    else
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_TASK_TO_DELETE_NOT_FOUND )
        }
    }

/* **************************************************************** *
 * RTDS_Sem_Info_Insert
 *-------------------------------------------------------------------
 * Inserts a semaphoreInfo struct in the semaphore info chained list
 *
 * PARAMETERS:
 *     semaphoreNumber : int                         Semaphore number representing its name
 *     semaphoreId : RTDS_SemaphoreId                ID of the semaphore
 *     RTDS_currentContext : RTDS_GlobalProcessInfo* CurrentContext of the task
 *
 * RETURN:
 *     RTDS_SemaphoreId    The Semaphore id of the inserted semaphore
 * **************************************************************** */
RTDS_SemaphoreId RTDS_Sem_Info_Insert(
 int semaphoreNumber ,
 RTDS_SemaphoreId semaphoreId ,
 RTDS_GlobalProcessInfo *RTDS_currentContext )
    {
    long semIsAvailable = -1;
    RTDS_GlobalSemaphoreInfo *semInfo;
    RTDS_GlobalSemaphoreInfo *newSemInfo;

    /* Allocate and fill in a new semInfo structure */
    newSemInfo = ( RTDS_GlobalSemaphoreInfo * )RTDS_MALLOC( sizeof( RTDS_GlobalSemaphoreInfo ) );

    newSemInfo->semaphoreNumber = semaphoreNumber;
    newSemInfo->next = NULL;

    /* Semaphore creation */
    newSemInfo->semaphoreId = semaphoreId;
    if ( newSemInfo->semaphoreId == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_INFO_INSERT )
        }

    /* Add the semaphore information to the chained list pointed by the RTDS_globalSemaphoreInfo global variable */
    RTDS_CRITICAL_SECTION_START;
    if ( RTDS_globalSemaphoreInfo == NULL )
        {
        RTDS_globalSemaphoreInfo = newSemInfo;
        }
    else
        {
        /* Let's get to the end of the list */
        semInfo = RTDS_globalSemaphoreInfo;
        while ( semInfo->next != NULL )
            {
            semInfo = semInfo->next;
            }
        semInfo->next = newSemInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;

    #if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER )
        semIsAvailable = RTDS_IsSemaphoreAvailable( semaphoreId );
    #endif
    RTDS_SIMULATOR_TRACE( RTDS_semaphoreCreated , semaphoreId , semIsAvailable , RTDS_currentContext );
    return semaphoreId;
    }

/* **************************************************************** *
 * RTDS_SemaphoreIdDelete
 *-------------------------------------------------------------------
 * Kills a semaphore and delete its info from the semaphore
 * information chained list and free the related memory
 *
 * PARAMETERS:
 *     semaphoreId : RTDS_SemaphoreId                The address of the semaphore to find the semaphore info block
 *     RTDS_currentContext : RTDS_GlobalProcessInfo* CurrentContext of the task
 * **************************************************************** */
void RTDS_SemaphoreIdDelete( RTDS_SemaphoreId semaphoreId , RTDS_GlobalProcessInfo *RTDS_currentContext )
    {
    RTDS_GlobalSemaphoreInfo *semInfo;
    RTDS_GlobalSemaphoreInfo *semInfoPrev;

    RTDS_SIMULATOR_TRACE( RTDS_semaphoreDeleted , semaphoreId , 0 , RTDS_currentContext );

    /* Remove the semaphore information from the chained list */
    semInfoPrev = NULL;
    RTDS_CRITICAL_SECTION_START;
    for ( semInfo = RTDS_globalSemaphoreInfo ; semInfo != NULL ; semInfo = semInfo->next )
        {
        if ( semInfo->semaphoreId == semaphoreId ) /* semaphore found */
            {
            if ( semInfoPrev == NULL )
                {
                RTDS_globalSemaphoreInfo = semInfo->next;
                }
            else
                {
                semInfoPrev->next = semInfo->next;
                }
            RTDS_SemDelete( semInfo->semaphoreId );
            RTDS_FREE( semInfo );
            break;
            }
        semInfoPrev = semInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    }

/* **************************************************************** *
 * RTDS_SemaphoreIdGIve
 *-------------------------------------------------------------------
 * Attempt to take a semaphore from its id
 *
 * PARAMETERS:
 *     semaphoreId : RTDS_SemaphoreId                The id of the semaphore
 *     RTDS_currentContext : RTDS_GlobalProcessInfo* CurrentContext of the task
 * **************************************************************** */
void RTDS_SemaphoreIdGive( RTDS_SemaphoreId semaphoreId , RTDS_GlobalProcessInfo *RTDS_currentContext )
    {
    RTDS_SIMULATOR_TRACE( RTDS_semGive , semaphoreId , 0 , RTDS_currentContext );
    if ( RTDS_SemGive( semaphoreId ) == RTDS_ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEMID_GIVE )
        }
    }

/* **************************************************************** *
 * RTDS_SemaphoreIdTake
 *-------------------------------------------------------------------
 * Attempt to take a semaphore from its id
 *
 * PARAMETERS:
 *     semaphoreId : RTDS_SemaphoreId               ID of the semaphore
 *     TIME_OUT : RTDS_SemaphoreTimeout             Timeout value   
 *     RTDS_currentContext : RTDS_GlobalProcessInfo CurrentContext of the task                           
 *
 * RETURNS:
 *     RTDS_SemaphoreStatus  Status
 * **************************************************************** */
RTDS_SemaphoreStatus RTDS_SemaphoreIdTake(
 RTDS_SemaphoreId semaphoreId ,
 RTDS_SemaphoreTimeout TIME_OUT ,
 RTDS_GlobalProcessInfo *RTDS_currentContext )
    {
    RTDS_SemaphoreStatus tmpStatus;
    long semIsAvailable = -1;
    
    RTDS_SIMULATOR_TRACE( RTDS_semTakeAttempt , semaphoreId , TIME_OUT , RTDS_currentContext );
    tmpStatus = RTDS_SemTake( semaphoreId , TIME_OUT );
    if ( tmpStatus == RTDS_OK )
        {
		    #if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER )
		        semIsAvailable=RTDS_IsSemaphoreAvailable( semaphoreId );
		    #endif
        RTDS_SIMULATOR_TRACE( RTDS_semTakeSucceded , semaphoreId , semIsAvailable , RTDS_currentContext );
        }
    else
        {
        RTDS_SIMULATOR_TRACE( RTDS_semTakeTimedOut , semaphoreId , 0 , RTDS_currentContext );
        }
    return tmpStatus;
    }

/* **************************************************************** *
 * RTDS_GetSemaphoreId
 *-------------------------------------------------------------------
 * Gets the id of a semaphore from its number (name)
 *
 * PARAMETERS:
 *     semaphoreNumber : int Semaphore number representing its name
 *
 * RETURNS:
 *     RTDS_SemaphoreId ID of the semaphore
 * **************************************************************** */
RTDS_SemaphoreId RTDS_GetSemaphoreId( int semaphoreNumber )
    {
    RTDS_GlobalSemaphoreInfo *semInfo;
    RTDS_SemaphoreId foundSemaphoreId;

    foundSemaphoreId = NULL;
    RTDS_CRITICAL_SECTION_START;
    for ( semInfo = RTDS_globalSemaphoreInfo ; semInfo != NULL ; semInfo = semInfo->next )
        {
        if ( semInfo->semaphoreNumber == semaphoreNumber ) /* semaphore found */
            {
            foundSemaphoreId = semInfo->semaphoreId;
            break;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;

    if ( foundSemaphoreId == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_SEMAPHORE_ID )
        }
    return foundSemaphoreId;    /* Might cause an RTOS exception if NULL*/
    }

/* **************************************************************** *
 * RTDS_TimerCleanUp
 *-------------------------------------------------------------------
 * Called when the last received message is a timer to figure out if
 * timer was not cancelled before it was received.
 *
 * PARAMETERS:
 *     currentContext : RTDS_GlobalProcessInfo* context of the current task
 * **************************************************************** */
void RTDS_TimerCleanUp(RTDS_GlobalProcessInfo * currentContext)
    {
    RTDS_TimerState * timer;
    RTDS_TimerState * prevTimer;
    
    prevTimer = NULL;
    for ( timer = currentContext->timerList ; timer != NULL; timer = timer->next )
      {
      /* If timer found and cancelled */
      if ( timer->timerUniqueId == currentContext->currentMessage->timerUniqueId )
          {
          if ( timer->state == RTDS_TIMER_CANCELLED )
              {
              /* Discard current message */
              RTDS_FREE( currentContext->currentMessage );
              currentContext->currentMessage = NULL;
              }

          /* Remove it from list of timers */
          if ( prevTimer == NULL )
              {
              currentContext->timerList = currentContext->timerList->next;
              }
          else
              {
              prevTimer->next = timer->next;
              }
          RTDS_FREE( timer );
          break;
          }
      prevTimer = timer;
      }
    }

/* **************************************************************** *
 * RTDS_TransitionCleanUp
 *-------------------------------------------------------------------
 * Called at the end of transitions:
 * - frees message buffer if valid
 * - re-organize save queue if state has changed
 *
 * PARAMETERS:
 *     RTDS_currentContext : RTDS_GlobalProcessInfo* CurrentContext of the task
 *     RTDS_sdlStatePrev : int                       Previous SDL state
 * **************************************************************** */
void RTDS_TransitionCleanUp( RTDS_GlobalProcessInfo *RTDS_currentContext , int RTDS_sdlStatePrev )
    {
    /* Free message buffer if valid */
    if ( RTDS_currentContext->currentMessage != NULL )
        {
        /* Free the message parameter structure if any */
        if (RTDS_currentContext->currentMessage->pData != NULL){
          RTDS_FREE(RTDS_currentContext->currentMessage->pData);
          }

        #ifdef RTDS_SIMULATOR
            /* Release the message unique id back to the pool */
            RTDS_ReleaseMessageUniqueId( RTDS_currentContext->currentMessage->messageUniqueId );
        #endif

        /* Free memory */
        RTDS_FREE( RTDS_currentContext->currentMessage );
        RTDS_currentContext->currentMessage = NULL;
        } /* End of if ( RTDS_currentContext->currentMessage != NULL ) */

    /* If SDL state has changed and messages have been saved: reorganise the save queue */
    if ( ( RTDS_currentContext->sdlState != RTDS_sdlStatePrev ) && ( RTDS_currentContext->writeSaveQueue != NULL ) )
        {
        RTDS_MessageHeader *message;
        /* Let's get to the end of the save queue */
        message = RTDS_currentContext->writeSaveQueue;
        while( message->next != NULL )
            {
            message = message->next;
            }
        message->next = RTDS_currentContext->readSaveQueue;
        RTDS_currentContext->readSaveQueue = RTDS_currentContext->writeSaveQueue;
        RTDS_currentContext->writeSaveQueue = NULL;
        }
    return;
    }

/* **************************************************************** *
 * RTDS_GetSystemTime
 *-------------------------------------------------------------------
 * Get system time
 *
 * RETURNS:
 *     System tick count value
 * **************************************************************** */
long RTDS_GetSystemTime( void )
    {
    return RTDS_TickGet();
    }

/* **************************************************************** *
 * RTDS_SetSystemTime
 *-------------------------------------------------------------------
 * Set the system time
 * 
 * PARAMETERS:
 *     timeValue : long  Time to be set
 * 
 * RETURNS:
 *     System tick count value
 * **************************************************************** */
#if defined( RTDS_SIMULATOR ) && defined( RTDS_SOCKET_PORT )
    void RTDS_SetSystemTime( long timeValue )
        {
        return RTDS_TickSet( timeValue );
        }
#endif

/* **************************************************************** *
 * **************************************************************** *
 * **************************************************************** *
 * THE CODE BELOW IS ONLY USED TO DEBUG WITH RTDS SDL-RT DEBUGGER
 * **************************************************************** *
 * **************************************************************** *
 * **************************************************************** */
#if defined( RTDS_SIMULATOR )
    void *RTDS_globalMessageUniqueIdPool = NULL;
#endif

#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACE ) || defined( RTDS_FORMAT_TRACE )
    RTDS_GlobalTraceInfo RTDS_globalTraceEntry = { RTDS_systemError , NULL , 0 };
    char *RTDS_globalPrintableParameters = NULL;
#endif
 
/********************************************************************
 * RTDS_GetMessageUniqueId
 *-------------------------------------------------------------------
 * Gets a message unique id for the simulator
 *
 * RETURNS:
 *     unsigned long The message unique id (minimum is 1)
 * **************************************************************** */
#if defined( RTDS_SIMULATOR )
    unsigned long RTDS_GetMessageUniqueId( void )
        {
        unsigned char *index;
        long uniqueByteId;
        long uniqueBitId;

        index = ( unsigned char * )RTDS_globalMessageUniqueIdPool;

        RTDS_CRITICAL_SECTION_START;
        for ( uniqueByteId = 0 ; uniqueByteId < RTDS_MESSAGE_UNIQUE_ID_POOL_SIZE ; uniqueByteId++ )
            {
            if ( *index != 0xFF )
                {
                for ( uniqueBitId = 0 ; uniqueBitId < 8 ; uniqueBitId++ )
                    {
                    if ( ( ( 1 << uniqueBitId ) & *index ) == 0 )
                        {
                        *index = *index | ( 1 << uniqueBitId );
                        RTDS_CRITICAL_SECTION_STOP;
                        return (8 * uniqueByteId + uniqueBitId + 1);
                        }
                     }
                }
            index++;
            }
        RTDS_CRITICAL_SECTION_STOP;

        /* All bits are set... No more message unique id */
        RTDS_SYSTEM_ERROR( RTDS_ERROR_NO_MORE_MSG_UNIQUE_ID )
        return 0;
        }
  
/********************************************************************
 * RTDS_ReleaseMessageUniqueId
 *-------------------------------------------------------------------
 * Make a message unique id available from the pool
 *
 * PARAMETERS:
 *     messageUniqueId  : unsigned long    Message unique id
 * **************************************************************** */
    void RTDS_ReleaseMessageUniqueId( unsigned long messageUniqueId )
        {
        unsigned char *index;

        if ( messageUniqueId == 0 ) /* probably a timer */
            {
            return;
            }
        messageUniqueId -= 1;
        index = ( unsigned char * )RTDS_globalMessageUniqueIdPool;
        index += ( unsigned char )( messageUniqueId / 8 );
        ( *index ) = ( *index ) & ~( 1 << messageUniqueId % 8 );
        }
#endif /* #if defined(RTDS_SIMULATOR) */
  
/********************************************************************
 * RTDS_SimulatorTrace
 *-------------------------------------------------------------------
 * Handles trace with the host depending on
 * the type of link available and options set in the profile.
 *
 * PARAMETERS:
 *     event : enum RTDS_EventType                  Trace event
 *     eventParameter1 : void *                     Parameter 1 of the trace
 *     eventParameter2 : long                       Parameter 2 of the trace
 *     currentContext : RTDS_GlobalProcessInfo *    Calling context
 * **************************************************************** */
/* Function defined is trace needed */
#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER )
    void RTDS_SimulatorTrace( 
     enum RTDS_EventType event ,
     void *eventParameter1 ,
     long eventParameter2 ,
     RTDS_GlobalProcessInfo *currentContext ,
     int waitAck )
        {
        #ifdef RTDS_FORMAT_TRACE
            unsigned long sysTime = 0;
            int formatStatus = RTDS_ERROR;
            char *formattedCmd = NULL;
            sysTime = RTDS_GetSystemTime();
        #endif /* RTDS_FORMAT_TRACE */

        /* Do not take semaphore when no ack is required
        i.e. when it comes from the debugger
        i.e. executed in the context of the task reading the socket
       otherwise dead lock ! */
       if ( waitAck == RTDS_DTRACE_ACK_WAIT )
           {
           RTDS_CRITICAL_TRACE_SECTION_START;
           }

       /* Trace information updated */
       RTDS_globalTraceEntry.event = event;
       RTDS_globalTraceEntry.eventParameter1 = ( void * )eventParameter1;
       RTDS_globalTraceEntry.eventParameter2 = ( long )eventParameter2;
       RTDS_globalTraceEntry.currentContext = ( RTDS_GlobalProcessInfo * )currentContext;

       #ifdef RTDS_FORMAT_TRACE
           formatStatus = RTDS_FormatTrace( sysTime , RTDS_globalTraceEntry , waitAck , &formattedCmd );
           if ( formattedCmd != NULL )
               {
               /* Send Trace to the external program: RTDS or MSC tracer */
               #if defined( RTDS_SOCKET_PORT )
                   RTDS_SOCKET_ACCESS_TAKE;
                   RTDS_SendSocket( globalClientSocketId , formattedCmd , strlen( formattedCmd ) );
                   RTDS_SOCKET_ACCESS_GIVE;
                if ( waitAck == RTDS_DTRACE_ACK_WAIT )
                    { 
                    RTDS_DTRACE_ACKNOWLEDGE_WAIT; 
                    }
               #endif
               }
           #ifdef RTDS_SIMULATOR
               RTDS_DummyTraceFunction();
           #endif
           RTDS_FREE( formattedCmd );
       #elif defined( RTDS_SIMULATOR )
           if ( ( event == RTDS_messageSent ) || ( event == RTDS_messageReceived ) )
               {
               RTDS_messageDataToString(
                &RTDS_globalPrintableParameters ,
                ( ( RTDS_MessageHeader * )eventParameter1 )->messageNumber ,
                ( ( RTDS_MessageHeader * )eventParameter1 )->dataLength ,
                ( void * )( ( ( RTDS_MessageHeader * )eventParameter1 )->pData ) ,
                RTDS_PARAM_CODEC_MAX_DEPTH );
               }
           RTDS_DummyTraceFunction();
           if ( ( event == RTDS_messageSent ) || ( event == RTDS_messageReceived ) )
               {
               RTDS_FREE( RTDS_globalPrintableParameters );
               RTDS_globalPrintableParameters = NULL;
               }
       #endif
       if ( waitAck == RTDS_DTRACE_ACK_WAIT )
           {
           RTDS_CRITICAL_TRACE_SECTION_STOP;
           }
       }
#endif /* defined(RTDS_SIMULATOR) || defined(RTDS_MSC_TRACER) */

/********************************************************************
 * RTDS_DummyTraceFunction
 *-------------------------------------------------------------------
 * The simulator sets a breakpoint on this function and reads the
 * RTDS_globalTraceEntry variable to see what happened
 * **************************************************************** */
#ifdef RTDS_SIMULATOR
    void RTDS_DummyTraceFunction( void )
        {
        #ifdef RTDS_BACK_TRACE_MAX_EVENT_NUM
            RTDS_TraceAdd();
        #endif
        }
#endif /* RTDS_SIMULATOR */


/* **************************************************************** *
 *    RTDS_DummyRunUntilQueueEmptyFunction
 * **************************************************************** *
 * As its name states... The simulator sets a breakpoint on this
 * function when the user asks tu run until the queue is empty. 
 * Whenever the system stops the breakpoint is removed.
 * **************************************************************** *
 * Parameters:
 *     None
 * Returns:
 *     nothing
 * **************************************************************** */
#ifdef RTDS_SIMULATOR
void RTDS_DummyRunUntilQueueEmptyFunction( void )
    {
    }
#endif /* RTDS_SIMULATOR */


/********************************************************************
 * RTDS_GetProcessInfoFromProcessId
 *-------------------------------------------------------------------
 * Returns the current RTDS_GlobalProcessInfo
 *
 * PARAMETERS:
 *     processId : RTDS_SdlInstanceId   ID of the task
 *
 * RETURNS:
 *     RTDS_GlobalProcessInfo *
 * **************************************************************** */
#if defined( RTDS_SIMULATOR )
    RTDS_GlobalProcessInfo * RTDS_GetProcessInfoFromProcessId( RTDS_SdlInstanceId *sdlInstanceId )
        {
        RTDS_GlobalProcessInfo *processInfo;

        if ( RTDS_globalProcessInfo == NULL )
            {
            return RTDS_globalProcessInfo;
            }
        for ( processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next )
            {
            if ( processInfo->mySdlInstanceId == sdlInstanceId )
               {
               return processInfo;
               }
            }
        return NULL; /* If not found */
        }
#endif /* defined(RTDS_SIMULATOR) */


/* **************************************************************** *
 *    RTDS_GetProcessInfoFromRtosTaskId
 * **************************************************************** *
 * Returns the current RTDS_GlobalProcessInfo
 * **************************************************************** *
 * Parameters:
 *     RTDS_RtosTaskId
 * Returns:
 *     RTDS_GlobalProcessInfo *
 * Use:
 *    presently used by RTDS_SYSTEM_ERROR macro
 * **************************************************************** */
#if defined( RTDS_SIMULATOR )
RTDS_GlobalProcessInfo * RTDS_GetProcessInfoFromRtosTaskId( RTDS_RtosTaskId rtosTaskId )
    {
    RTDS_GlobalProcessInfo * processInfo;

    if ( RTDS_globalProcessInfo == NULL )
        {
        return RTDS_globalProcessInfo;
        }
    for ( processInfo = RTDS_globalProcessInfo; processInfo != NULL; processInfo = processInfo->next )
        {
        if ( processInfo->myRtosTaskId == rtosTaskId )
            {
            return processInfo;
            }
        }

    return NULL; /* If not found */
    }
#endif /* defined(RTDS_SIMULATOR) */

/********************************************************************
 * RTDS_GetProcessInfoFromProcessNum
 *-------------------------------------------------------------------
 * Returns the RTDS_GlobalProcessInfo of the corresponding process number
 *
 * PARAMETERS:
 *     processNumber : int    Name of the process
 *
 * RETURNS:
 *     RTDS_GlobalProcessInfo*
 * **************************************************************** */
#if defined( RTDS_FORMAT_TRACE ) || defined( RTDS_SIMULATOR )
    RTDS_GlobalProcessInfo * RTDS_GetProcessInfoFromProcessNum( int processNumber )
        {
        RTDS_GlobalProcessInfo *processInfo;

        if ( RTDS_globalProcessInfo == NULL )
            {
            return RTDS_globalProcessInfo;
            }
        for ( processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next )
            {
            if ( processInfo->sdlProcessNumber == processNumber )
                {
                return processInfo;
                }
            }
        return NULL; /* If not found */
        }
#endif

/********************************************************************
 * RTDS_GetCurrentProcessId
 *-------------------------------------------------------------------
 * Returns the current RTDS_RtosTaskId
 *
 * RETURNS:
 *     RTDS_RtosTaskId
 * **************************************************************** */
#if defined( RTDS_SIMULATOR ) 
    RTDS_RtosTaskId RTDS_GetCurrentProcessId( void )
        {
        return RTDS_GetCurrentThread();
        }
#endif 
  
/********************************************************************
 * RTDS_IsSemaphoreAvailable
 *-------------------------------------------------------------------
 * Returns the semaphore state
 *
 * PARAMETERS:
 *     semaphoreId : RTDS_SemaphoreId  ID of the semaphore
 *
 * RETURNS:
 *     long : -1 if function not available
 *             0 if semaphore not available
 *             1 if semaphore available or n for counter number
 * **************************************************************** */    
#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER )
    long RTDS_IsSemaphoreAvailable( RTDS_SemaphoreId semaphoreId )
        {
        int stillAvailable = -1;
        if ( RTDS_SemTake( semaphoreId , RTDS_SEMAPHORE_TIME_OUT_NO_WAIT ) == RTDS_TIMEOUT )
            {
            stillAvailable = 0;
            }
        else
            {
            if ( RTDS_SemGive( semaphoreId ) == RTDS_ERROR )
                {
                RTDS_SYSTEM_ERROR(RTDS_ERROR_IS_SEM_AVAILABLE)
                }
            stillAvailable = 1;
            }
        return stillAvailable;
        }
#endif

/********************************************************************
 * RTDS_SimulatorMsgQueueSend
 *-------------------------------------------------------------------
 * Send a message in a process's queue without calling RTDS_SIMULATOR_TRACE
 *
 * PARAMETERS:
 *     messageNumber : long                          MessageNumber representing a message name
 *     dataLength : long                             Length of data pointed by pData
 *     pData : unsigned char*                        Pointer on data sent with the message
 *     receiver : RTDS_QueueId                       Message receiver queue address
 *     sender : RTDS_QueueId                         Message sender queue address
 *     RTDS_currentContext : RTDS_GlobalProcessInfo* Current context of the task
 * **************************************************************** */
#if defined( RTDS_SIMULATOR ) && defined( RTDS_SOCKET_PORT )
    void RTDS_SimulatorMsgQueueSend(
      long messageNumber ,
      long dataLength ,
      unsigned char *pData ,
      RTDS_SdlInstanceId *receiver ,
      RTDS_SdlInstanceId *sender ,
      RTDS_GlobalProcessInfo *RTDS_currentContext )
        {
        RTDS_MessageHeader RTDS_messageToSend;

        RTDS_messageToSend.messageNumber = messageNumber;
        RTDS_messageToSend.timerUniqueId = 0;
        RTDS_messageToSend.sender = sender;
        RTDS_messageToSend.receiver = receiver;
        RTDS_messageToSend.dataLength = dataLength;
        RTDS_messageToSend.pData = pData;
        RTDS_messageToSend.next = NULL;

        RTDS_messageToSend.messageUniqueId = RTDS_GetMessageUniqueId();
        RTDS_SimulatorTrace( RTDS_messageSent , ( void * )&RTDS_messageToSend , ( long )receiver , RTDS_currentContext , RTDS_DTRACE_ACK_NOWAIT );

        if ( RTDS_MsgQSend( receiver->queueId , &RTDS_messageToSend ) == RTDS_ERROR )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_SEND )
            }
        }
#endif

/********************************************************************
 * RTDS_forceTimer
 *-------------------------------------------------------------------
 * Call RTDS_WatchDogFunction(RTDS_TimerState * parameter)
 * Function called by TCP client to force timer to go off.
 *
 * PARAMETERS:
 *     sdlInstanceId : RTDS_SdlInstanceId   * sdlInstanceId
 *     timerNumber :   int                    TimerName
 * **************************************************************** */
#if defined( RTDS_SIMULATOR ) && defined( RTDS_SOCKET_PORT )
    void RTDS_ForceTimer( RTDS_SdlInstanceId *sdlInstanceId , int timerNumber )
        {
        RTDS_GlobalProcessInfo *processInfo;
        RTDS_TimerState *timerInfo;
        RTDS_TimerState *timerState = NULL;
        long timeOut;

        RTDS_CRITICAL_SECTION_START;
        /* Find timerState (RTDS timer ID) from its timer number and the related sdlInstanceId */
        for ( processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next )
            {
            if ( processInfo->mySdlInstanceId == sdlInstanceId )
                {
                for ( timerInfo = processInfo->timerList ; timerInfo != NULL ; timerInfo = timerInfo->next )
                    {
                    if ( ( timerInfo->timerNumber == timerNumber ) && ( timerInfo->state == RTDS_TIMER_OK ) )
                        {
                        timerState = timerInfo;
                        timeOut = timerState->timeoutValue;
                        break;
                        }
                    }
                }
            }
        RTDS_CRITICAL_SECTION_STOP;

        /* Call timer function to send SDL message */
        if( timerState != NULL )
            {
            /* Let's try to cancel the watchdog */
            if ( RTDS_TimerDelete( timerState ) == RTDS_ERROR )
                {
                RTDS_SYSTEM_ERROR( RTDS_ERROR_TIMER_DELETE )
                }
            RTDS_WatchDogFunction( timerState );
            #ifdef RTDS_DISCRETE_TIME
                RTDS_TickSet( timeOut );
            #endif /* RTDS_DISCRETE_TIME */
            }
        }
#endif /* #if defined(RTDS_SIMULATOR) && defined(RTDS_SOCKET_PORT) */
