/* PragmaDev RTDS ThreadX integration */

#include "RTDS_MACRO.h"

/* **************************************************************** *
 * List of functions defined in this file
 * **************************************************************** *
 *	RTDS_GetTimerUniqueId
 *	RTDS_WatchDogFunction
 *	RTDS_StartTimer
 *	RTDS_StopTimer
 *	RTDS_GetProcessQueueId
 *	RTDS_MsgQueueSend
 *	RTDS_SimulatorMsgQueueSend
 *	RTDS_MsgQueueReceive
 *	RTDS_ProcessCreate
 *	RTDS_ProcessKill
 *	RTDS_BinarySemaphoreCreate
 *	RTDS_MutexSemaphoreCreate
 *	RTDS_CountingSemaphoreCreate
 *	RTDS_Sem_Info_Insert
 *	RTDS_Sem_Delete
 *	RTDS_SemaphoreIdTake
 *	RTDS_GetSemaphoreId
 *	RTDS_GetSystemTime
 *	RTDS_GetMessageUniqueId
 *	RTDS_ReleaseMessageUniqueId
 *	RTDS_DummyTraceFunction
 * **************************************************************** */


/* **************************************************************** *
 *	RTDS_GetTimerUniqueId
 * **************************************************************** *
 * Get an available timer unique id.
 * Is used to set timers
 * **************************************************************** *
 * Parameters:
 *	 timerList: points to the first element of
 * the chained list of RTDS_TimerState
 * Returns:
 *	 The unique timer Id
 * **************************************************************** *
 * Context:
 *	 The chained list of RTDS_TimerState is sorted by increasing
 *	 unique timer id
 *	 timerUniqueId = NULL means the message is not a timer; so it
 *	 is avoided.
 * **************************************************************** */

long RTDS_GetTimerUniqueId(RTDS_TimerState *timerList)
	{
	RTDS_TimerState *RTDS_prevTimer,*RTDS_timer;
	long newTimerId;

	RTDS_CRITICAL_SECTION_START;
	/* If list is empty take 1 */
	if ( timerList == NULL )
	newTimerId = 1;
	/* If 1 is available, take it */
	else if ( timerList->timerUniqueId != 1 )
	newTimerId = 1;
	else
	{
	/* If there is a gap in the timerId chained list, take an Id in the gap */
	RTDS_prevTimer = timerList;
	newTimerId = 0;
	for ( RTDS_timer = timerList->next ; RTDS_timer != NULL; RTDS_timer = RTDS_timer->next )
		{
		if ( RTDS_timer->timerUniqueId != RTDS_prevTimer->timerUniqueId + 1 ) newTimerId = RTDS_prevTimer->timerUniqueId + 1;
		RTDS_prevTimer = RTDS_timer;
		}
	/* No gap, let's take the next value */
	if ( newTimerId == 0 ) newTimerId = RTDS_prevTimer->timerUniqueId + 1;
	}

	RTDS_CRITICAL_SECTION_STOP;
	/* Check the counter did not go back to 0 */
	if ( newTimerId == 0 )
	RTDS_SYSTEM_ERROR(RTDS_ERROR_NO_MORE_TIMER_UNIQUE_ID);
	return newTimerId;
	}



/* **************************************************************** *
 *	RTDS_WatchDogFunction
 * **************************************************************** *
 * Send a timer message using the parameters given to the watchdog
 * **************************************************************** *
 * Parameters:
 *	 parameter: Needed parameters to create and send the timer:
 *		 - QueueId of the receiver
 *		 - TimerNumber
 *		 - TimerUniqueId
 *		 - WatchDogId to destroy it
 * Returns:
 *	 Nothing
 * **************************************************************** *
 * Context:
 *	 Each time a timer is set, a watchdog is created; that's why it
 *	 has to destroy itself.
 * **************************************************************** */


void RTDS_WatchDogFunction(ULONG parameter)
	{
	UINT	status;

	/* Send the message without waiting */
	status = tx_queue_send(
		((RTDS_TimerState * )parameter)->receiverId->queueId,
		(void *)&(((RTDS_TimerState * )parameter)->messageAddress),
		TX_NO_WAIT );

	if ( status != TX_SUCCESS )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_MSG_Q_SEND_IN_WATCHDOG);

	/* Delete the watchdog */
	status = tx_timer_deactivate(((RTDS_TimerState * )parameter)->watchDogId);
	if ( status != TX_SUCCESS )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_DEACTIVATING_TIMER);
	}



/* **************************************************************** *
 *	RTDS_StartTimer
 * **************************************************************** *
 * Starts a watchdog with the necessary parameters to create the
 * timer when it goes off.
 * **************************************************************** *
 * Parameters:
 *	- instanceId of the receiver
 *	- TimerNumber
 *	- TimerUniqueId
 *	- delay of the timer
 *	- timerStateList: Address of the list of timer
 *	- currentContext (RTDS_GlobalProcessInfo *): active task context
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_StartTimer(
  RTDS_SdlInstanceId      * instanceId,
	long timerNumber,
	long timerUniqueId,
	int delay,
	RTDS_TimerState **pTimerStateList,
	RTDS_GlobalProcessInfo *RTDS_currentContext)
	{
	TX_TIMER						*watchDogId;
	RTDS_TimerState 		*timerState, *previousTimerState, *newTimerState;
	RTDS_MessageHeader	*timerMessage;

	/* Allocate the message now because it is not possible in the application timer function... */
	timerMessage = (RTDS_MessageHeader *) RTDS_MALLOC(sizeof(RTDS_MessageHeader));

	/* Fill in the message parameters */
	#ifdef RTDS_SIMULATOR
	timerMessage->messageUniqueId = 0;
	#endif
	timerMessage->messageNumber = timerNumber;
	timerMessage->timerUniqueId = timerUniqueId;
	timerMessage->sender = instanceId;
	timerMessage->dataLength = 0;
	timerMessage->pData = NULL;

	/* Update the list of timers */
	/* Important note: it is a sorted list based on the timerUniqueId field */
	newTimerState = (RTDS_TimerState *)RTDS_MALLOC(sizeof(RTDS_TimerState));
	if (newTimerState == NULL)
	RTDS_SYSTEM_ERROR(RTDS_ERROR_MALLOC_TIMER_STATE_IN_START_TIMER);

	watchDogId = (TX_TIMER *)RTDS_MALLOC( sizeof(TX_TIMER) );
	/* Create a new watchdog */
	if ( tx_timer_create(
			watchDogId,
			"RTDS_Timer",
			RTDS_WatchDogFunction,
			(ULONG)newTimerState,
			(ULONG)delay,
			(ULONG)0,
			TX_NO_ACTIVATE) != TX_SUCCESS)
		RTDS_SYSTEM_ERROR(RTDS_ERROR_WATCHDOG_CREATION);

	/* Initialize the new element */
	newTimerState->state		= RTDS_TIMER_OK;
	newTimerState->timerNumber	= timerNumber;
	newTimerState->timeoutValue = delay + RTDS_GetSystemTime(); /* Only way I found to follow up the timer */
	newTimerState->timerUniqueId	= timerUniqueId;
	newTimerState->next 	= NULL;
	newTimerState->watchDogId 	= watchDogId;
	newTimerState->receiverId = instanceId;
	newTimerState->messageAddress = timerMessage;

	/* Insert the new element */
	if (*pTimerStateList == NULL) /* The list is empty */
	*pTimerStateList = newTimerState;
	else	/* The list is not empty */
	{
	previousTimerState = NULL;
	for (timerState = *pTimerStateList ; timerState != NULL ; timerState = timerState->next)
		{
		if (timerState->timerUniqueId > timerUniqueId)
		{
		if (previousTimerState == NULL)
			{
			*pTimerStateList = newTimerState;
			newTimerState->next = timerState;
			}
		else
			{
			previousTimerState->next = newTimerState;
			newTimerState->next = timerState;
			}
		break;
		}
		previousTimerState = timerState;
		}
	if (timerState == NULL) /* Inserted at the end of the list */
		previousTimerState->next = newTimerState;
	}


	RTDS_SIMULATOR_TRACE(RTDS_timerStarted, newTimerState, delay, RTDS_currentContext);

	/* Starts the watchdog */
	if ( tx_timer_activate(watchDogId)	!= TX_SUCCESS )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_WATCHDOG_START);

}



/* **************************************************************** *
 *	RTDS_StopTimer
 * **************************************************************** *
 * Stops a timer in trying to delete the watchdog. If unsuccessfull
 * set it cancelled in the timer chained list verified by the RTDS
 * kernel
 * **************************************************************** *
 * Parameters:
 *	- TimerNumber
 *	- pTimerStateList pointing to the timer chained list
 *	- RTDS_currentContext currentContext of the task
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_StopTimer(
	long timerNumber,
	RTDS_TimerState **pTimerStateList,
	RTDS_GlobalProcessInfo *RTDS_currentContext )
	{
	RTDS_TimerState  *RTDS_timer, *RTDS_prevTimer;
	RTDS_MessageHeader *RTDS_message, *RTDS_prevMessage;

	RTDS_prevTimer = NULL;
	for ( RTDS_timer = *pTimerStateList ; RTDS_timer != NULL; RTDS_timer = RTDS_timer->next )
		{
		if (( RTDS_timer->timerNumber == timerNumber ) && ( RTDS_timer->state != RTDS_TIMER_CANCELLED ))
			{
			RTDS_SIMULATOR_TRACE(RTDS_timerCancelled, RTDS_timer, NULL, RTDS_currentContext);

			/* Let's try to cancel the watchdog */
			if ( tx_timer_delete(RTDS_timer->watchDogId) == TX_SUCCESS )
				{
				/* Release the timer message buffer */
				RTDS_FREE(RTDS_timer->messageAddress);

				/* Remove it from the list */
				/* Is the first of the list */
				if (RTDS_prevTimer == NULL)
					{
					*pTimerStateList = RTDS_timer->next;
					}
				/* Not the first of the list */
				else
					{
					RTDS_prevTimer->next = RTDS_timer->next;
					}
				RTDS_FREE(RTDS_timer);
				}
			/* Could not cancel the timer. Probably went off allready */
			/* Set it cancelled in the list so that it is ignored when received */
			else
				{
				RTDS_timer->state = RTDS_TIMER_CANCELLED;
				}
			return;
			}
		RTDS_prevTimer = RTDS_timer;
		}

	/* If execution gets here: the timer might be in the save queue */
	RTDS_prevMessage = NULL;
	for ( RTDS_message=RTDS_currentContext->readSaveQueue ; RTDS_message!=NULL ; RTDS_message=RTDS_message->next )
		{
		if ( RTDS_message->messageNumber == timerNumber )
			{
			/* Remove it from the list */
			/* Is the first of the list */
			if (RTDS_prevMessage == NULL)
				{
				RTDS_currentContext->readSaveQueue = RTDS_message->next;
				}
			/* Not the first of the list */
			else
				{
				RTDS_prevMessage->next = RTDS_message->next;
				}
			RTDS_FREE(RTDS_message);
			return;
			}
		RTDS_prevMessage = RTDS_message;
		}

	/* If execution gets here: the timer might be in the save queue */
	RTDS_prevMessage = NULL;
	for ( RTDS_message=RTDS_currentContext->writeSaveQueue ; RTDS_message!=NULL ; RTDS_message=RTDS_message->next )
		{
		if ( RTDS_message->messageNumber == timerNumber )
			{
			/* Remove it from the list */
			/* Is the first of the list */
			if (RTDS_prevMessage == NULL)
				{
				RTDS_currentContext->writeSaveQueue = RTDS_message->next;
				}
			/* Not the first of the list */
			else
				{
				RTDS_prevMessage->next = RTDS_message->next;
				}
			RTDS_FREE(RTDS_message);
			return;
			}
		RTDS_prevMessage = RTDS_message;
		}

	}



/* **************************************************************** *
 *	RTDS_GetProcessQueueId
 * **************************************************************** *
 * Returns the queueId of process based on its name
 * **************************************************************** *
 * Parameters:
 *		 - process name as a number defined in RTDS_gen.h
 * Returns:
 *	 the process queue id if found
 * Error:
 *	 System error call if not found and NULL returned
 * **************************************************************** */

RTDS_QueueId RTDS_GetProcessQueueId(int processNumber)
	{
	RTDS_GlobalProcessInfo	*processInfo;
	RTDS_QueueId			foundQueue = NULL;

	RTDS_CRITICAL_SECTION_START;
	for (processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next)
		{
		if ( processInfo->sdlProcessNumber == processNumber )
			{
			foundQueue = processInfo->mySdlInstanceId;
			break;
			}
		}
	RTDS_CRITICAL_SECTION_STOP;

	if (foundQueue == NULL)
		RTDS_SYSTEM_ERROR(RTDS_ERROR_GET_PROCESS_QUEUE_ID);

	return foundQueue;
	}



/* **************************************************************** *
 *	RTDS_MsgQueueSend
 * **************************************************************** *
 * Send a message in a process's queue
 * **************************************************************** *
 * Parameters:
 *	- messageNumber representing a message name
 *	- dataLength length of data pointed by pData
 *	- pData  pointer on data sent with the message
 *	- receiver message receiver queue address
 *	- sender	 message sender queue address
 *	- RTDS_currentContext currentContext of the task
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_MsgQueueSend(
	long										messageNumber,
	long										dataLength,
	unsigned char 					*pData,
	RTDS_QueueId						receiver,
	RTDS_QueueId						sender,
	RTDS_GlobalProcessInfo	*RTDS_currentContext )
	{
	RTDS_MessageHeader	*RTDS_messageToSend;

	RTDS_messageToSend = (RTDS_MessageHeader *) RTDS_MALLOC(sizeof(RTDS_MessageHeader));

	RTDS_messageToSend->messageNumber = messageNumber;
	RTDS_messageToSend->timerUniqueId = 0;
	RTDS_messageToSend->sender = sender;
	RTDS_messageToSend->dataLength = dataLength;
	RTDS_messageToSend->pData = pData;
	RTDS_messageToSend->next = NULL;
	#ifdef RTDS_SIMULATOR
		RTDS_messageToSend->messageUniqueId = RTDS_GetMessageUniqueId();
		RTDS_messageDataToString(&RTDS_globalPrintableParameters, messageNumber, dataLength, (void *)pData, RTDS_PARAM_CODEC_MAX_DEPTH);
		RTDS_SIMULATOR_TRACE(RTDS_messageSent, RTDS_messageToSend, receiver, RTDS_currentContext);
		RTDS_FREE(RTDS_globalPrintableParameters);
		RTDS_globalPrintableParameters = NULL;
	#endif

	if ( tx_queue_send(receiver->queueId, (VOID *)&RTDS_messageToSend, TX_WAIT_FOREVER) != TX_SUCCESS )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_MSG_QUEUE_SEND);
	}



/* **************************************************************** *
 *	RTDS_MsgQueueReceive
 * **************************************************************** *
 * Receives a message from a process's queue
 * Note the memory has been allocated by the sender of the message
 * Only a pointer is received
 * **************************************************************** *
 * Parameters:
 *		 - queue id to read the messsage from
 *		 - pointer on the message to receive
 *		 - sier of message
 *		 - mode used to read (blocking or non blocking)
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_MsgQueueReceive(RTDS_SdlInstanceId * instanceId, char **message, int size, int mode)
	{
	if ( tx_queue_receive(instanceId->queueId, message, mode) != TX_SUCCESS )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_MSG_QUEUE_RECEIVE);
	}




/* **************************************************************** *
 *	RTDS_ProcessCreate
 * **************************************************************** *
 * Create a new SDL process and add a processInfo struct in the
 * process info chained list
 * **************************************************************** *
 * Parameters:
 *	- process name
 *	- process number
 *	- address of the process function
 *	- priority fo the process
 *	- the address of the address of the msg queue to be created
 *	- the address of the msg queue of the parent process
 *	- RTDS_currentContext currentContext of the task
 * Returns:
 *	 Nothing but updates the OFFSPRING of the caller
 * **************************************************************** */

void RTDS_ProcessCreate(
	char										*processName,
	int 										processNumber,
	void										(*functionAddress)(ULONG),
	int 										priority,
	RTDS_SdlInstanceId			**pOffspringId,
	RTDS_SdlInstanceId			*selfQueueId,
	RTDS_GlobalProcessInfo	*RTDS_currentContext )
	{
	RTDS_GlobalProcessInfo	*processInfo, *newProcessInfo, *previousProcessInfo;
	VOID										*stack_start;
	UINT										retCode;


	/* first of all: is there any thread to delete ? */
	/* This happens because a thread can not delete itself... */
	previousProcessInfo = NULL;
	RTDS_CRITICAL_SECTION_START;
	for (processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next)
		{
		/* A stack pointer set to NULL means the thread must be deleted... */
		if ( processInfo->pStack == NULL )	/* Found ! */
			{
			/* Update the process information chained list */
			if ( previousProcessInfo == NULL )	/* First one in the list */
				RTDS_globalProcessInfo = processInfo->next;
			else
				previousProcessInfo->next = processInfo->next;

			/* Delete the thread */
			retCode = tx_thread_delete(processInfo->myRtosTaskId);
			if ( retCode != TX_SUCCESS )
				RTDS_SYSTEM_ERROR(RTDS_ERROR_PROCESS_CREATE_THREAD_DELETE);
			/* Free the ThreadX thread control block */
			RTDS_FREE(processInfo->myRtosTaskId);
			RTDS_FREE(processInfo);
			}
		else
			previousProcessInfo = processInfo;
		}
	RTDS_CRITICAL_SECTION_STOP;


	/* Allocate and fill in a new processInfo structure */
	newProcessInfo = (RTDS_GlobalProcessInfo *)RTDS_MALLOC(sizeof(RTDS_GlobalProcessInfo));
	if ( newProcessInfo == NULL )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_PROCESS_CREATE_MALLOC);

	/* Sets the process name as a number */
	newProcessInfo->sdlProcessNumber = processNumber;
	/* SDL initial state */
	newProcessInfo->sdlState = 0;

	/* Last structure of the chained list */
	newProcessInfo->next							     = NULL;
	newProcessInfo->parentSdlInstanceId    = selfQueueId;
	newProcessInfo->offspringSdlInstanceId = NULL;
	newProcessInfo->currentMessage		     = NULL;
	newProcessInfo->timerList 				     = NULL;
	#ifdef RTDS_SIMULATOR
	newProcessInfo->priority					     = priority;
	#endif

	/* New task's queue creation */

  *pOffspringId = ( RTDS_SdlInstanceId * )RTDS_MALLOC( sizeof( RTDS_SdlInstanceId ) );
  ( *pOffspringId )->instanceNumber = 0;
  newProcessInfo->mySdlInstanceId = *pOffspringId;

	/* Allocate memory for message queue. To be customized for the target. */
	newProcessInfo->pQueue = (void *)RTDS_MALLOC(RTDS_QUEUE_MAX_MSG_LENGTH*RTDS_QUEUE_MAX_MSG);
	/* Memory needs to be allocated for the queue control block */
	( *pOffspringId )->queueId = (RTDS_RtosQueueId) RTDS_MALLOC( sizeof(TX_QUEUE) );
	retCode = tx_queue_create(
			( *pOffspringId )->queueId,
			processName,
			RTDS_QUEUE_MAX_MSG_LENGTH,
			newProcessInfo->pQueue,
			(ULONG)RTDS_QUEUE_MAX_MSG_LENGTH*RTDS_QUEUE_MAX_MSG);
	if ( retCode != TX_SUCCESS )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_PROCESS_CREATE_MSG_Q_CREATE);

	newProcessInfo->mySdlInstanceId = *pOffspringId;

	/* Task creation but not started */
	/* Allocate memory for thread stack. To be customized. */
	stack_start = (void *)RTDS_MALLOC(RTDS_TASK_STACK_SIZE);
	newProcessInfo->pStack = stack_start;
	/* Memory needs to be allocated for the thread control block */
	newProcessInfo->myRtosTaskId = (RTDS_RtosTaskId) RTDS_MALLOC( sizeof(TX_THREAD) );
	retCode = tx_thread_create(
			newProcessInfo->myRtosTaskId,
			processName,
			functionAddress,
			(ULONG) newProcessInfo,
			stack_start,
			(ULONG) RTDS_TASK_STACK_SIZE,
			(UINT) priority,
			(UINT) priority,
			TX_NO_TIME_SLICE,
			TX_DONT_START);
	if ( retCode != TX_SUCCESS )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_PROCESS_CREATE_TASK_SPAWN);

	/* Add the process information to the chained list pointed by the RTDS_globalProcessInfo global variable */
	RTDS_CRITICAL_SECTION_START;
	if ( RTDS_globalProcessInfo == NULL )
		RTDS_globalProcessInfo = newProcessInfo;
	else
		{
		/* Let's get to the end of the list */
		for (processInfo = RTDS_globalProcessInfo ; processInfo->next != NULL ; processInfo = processInfo->next) ;
		processInfo->next = newProcessInfo;
		}
	RTDS_CRITICAL_SECTION_STOP;

	RTDS_SIMULATOR_TRACE(RTDS_processCreated, newProcessInfo, NULL, RTDS_currentContext);

	/* The newly created task can now run: RTDS_globalProcessInfo and trace are up to date. */
	retCode = tx_thread_resume(newProcessInfo->myRtosTaskId);
	if ( retCode != TX_SUCCESS )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_PROCESS_CREATE_RESUME);
	}




/* **************************************************************** *
 *	RTDS_ProcessKill
 * **************************************************************** *
 * Kills an SDL process and delete its queue and process info block
 * **************************************************************** *
 * Parameters:
 *	- the address of the process current context
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_ProcessKill( RTDS_GlobalProcessInfo  *RTDS_currentContext )
	{
	RTDS_GlobalProcessInfo	*processInfo;
	RTDS_TimerState 				*RTDS_timer;
	RTDS_MessageHeader			*message, *tmpMessage;
	RTDS_RtosTaskId					pidToKill=NULL;

	/* Let's free the current message if any */
	if (RTDS_currentContext->currentMessage != NULL )
		{
		#ifdef RTDS_SIMULATOR
			/* Release the message unique id back to the pool */
			RTDS_ReleaseMessageUniqueId(RTDS_currentContext->currentMessage->messageUniqueId);
		#endif

    /* Free the message parameter structure if any */
    if (RTDS_currentContext->currentMessage->pData != NULL){
      RTDS_FREE(RTDS_currentContext->currentMessage->pData);
      }

		/* Free memory */
		RTDS_FREE(RTDS_currentContext->currentMessage);
		}

	/* Let's clean the timer chained list to free memory and stop watchdogs */
	for ( RTDS_timer = RTDS_currentContext->timerList ; RTDS_timer != NULL ; RTDS_timer = RTDS_currentContext->timerList)
		{
		tx_timer_delete(RTDS_timer->watchDogId);
		RTDS_currentContext->timerList = RTDS_timer->next;
		RTDS_FREE(RTDS_timer);
		}

	/* Clean the save queue: free messages and message unique ids in the read and write save queues */
	for (message=RTDS_currentContext->readSaveQueue;message!=NULL;)
		{
		#ifdef RTDS_SIMULATOR
			RTDS_ReleaseMessageUniqueId(message->messageUniqueId);
		#endif
		tmpMessage = message;
		message = message->next;
		RTDS_FREE(tmpMessage);
		}

	for (message=RTDS_currentContext->writeSaveQueue;message!=NULL;)
		{
		#ifdef RTDS_SIMULATOR
			RTDS_ReleaseMessageUniqueId(message->messageUniqueId);
		#endif
		tmpMessage = message;
		message = message->next;
		RTDS_FREE(tmpMessage);
		}

	RTDS_CRITICAL_SECTION_START;
	for (processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next)
		{
		/* The queue id is used to find the process id */
		if ( processInfo->mySdlInstanceId == RTDS_currentContext->mySdlInstanceId )  /* Found ! */
			{
			RTDS_SIMULATOR_TRACE(RTDS_processDied, processInfo, NULL, RTDS_currentContext);

			pidToKill = processInfo->myRtosTaskId;
			/* Delete the message queue */
			if ( tx_queue_delete(processInfo->mySdlInstanceId->queueId) != TX_SUCCESS )
				RTDS_SYSTEM_ERROR(RTDS_ERROR_MSG_QUEUE_DELETE);
			/* Free the queue itself */
			RTDS_FREE(processInfo->pQueue);
			/* Free the ThreadX queue control block */
			RTDS_FREE(processInfo->mySdlInstanceId->queueId);
			/* Free the thread stack */
			RTDS_FREE(processInfo->pStack);
			/* We will use the fact that the stack is NULL to free the thread control block during the next thread creation... */
			processInfo->pStack = NULL;
			/* Put an undefined state to show the process is dead */
			processInfo->sdlState = -1;
			/* Note processInfo and processInfo->processId will be free during the next thread creation */
			break;
			}
		}

	RTDS_CRITICAL_SECTION_STOP;

	/* Terminate the thread */
	/* Note the thread can not delete itself so it will be done in the next thread creation context... */
	if ( pidToKill != NULL)
		{
		if ( tx_thread_terminate(pidToKill) != TX_SUCCESS )
			RTDS_SYSTEM_ERROR(RTDS_ERROR_TASK_DELETE);
		}
	else
		RTDS_SYSTEM_ERROR(RTDS_ERROR_TASK_TO_DELETE_NOT_FOUND);

	}




/* **************************************************************** *
 *	RTDS_BinarySemaphoreCreate
 * **************************************************************** *
 * Creates a mutex semaphore
 * **************************************************************** *
 * Parameters:
 *	 - semaphore name
 *	 - semaphore number
 *	 - semaphore initial state
 *	 - RTDS_currentContext currentContext of the task
 * Returns:
 *	 - semaphoreId
 * **************************************************************** */

RTDS_SemaphoreId	RTDS_BinarySemaphoreCreate(char *name, int semaphoreNumber, unsigned char initialState, RTDS_GlobalProcessInfo	*RTDS_currentContext)
	{
	RTDS_SemaphoreId	newSemaphore;

	newSemaphore = (TX_SEMAPHORE *) RTDS_MALLOC(sizeof(TX_SEMAPHORE));

	if ( tx_semaphore_create(newSemaphore, name, initialState) != TX_SUCCESS )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_BINARY_SEMAPHORE_CREATE);
	RTDS_Sem_Info_Insert( semaphoreNumber, newSemaphore, RTDS_binary, RTDS_currentContext );

	return newSemaphore;
	}



/* **************************************************************** *
 *	RTDS_MutexSemaphoreCreate
 * **************************************************************** *
 * Creates a mutex semaphore
 * **************************************************************** *
 * Parameters:
 *	 - semaphore name
 *	 - semaphore number
 *	 - RTDS_currentContext currentContext of the task
 * Returns:
 *	 - semaphoreId
 * **************************************************************** */

RTDS_MutexId RTDS_MutexSemaphoreCreate(char *name, int semaphoreNumber, RTDS_GlobalProcessInfo	*RTDS_currentContext)
	{
	RTDS_MutexId	newSemaphore;

	newSemaphore = (TX_MUTEX *) RTDS_MALLOC(sizeof(TX_MUTEX));

	if ( tx_mutex_create(newSemaphore, name, TX_NO_INHERIT) != TX_SUCCESS )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_MUTEX_SEMAPHORE_CREATE);
	RTDS_Sem_Info_Insert( semaphoreNumber, (RTDS_SemaphoreId)newSemaphore, RTDS_mutex, RTDS_currentContext);

	return newSemaphore;
	}



/* **************************************************************** *
 *	RTDS_CountingSemaphoreCreate
 * **************************************************************** *
 * Creates a counting semaphore
 * **************************************************************** *
 * Parameters:
 *	 - semaphore name
 *	 - semaphore number
 *	 - initial semaphore count value
 *	 - RTDS_currentContext currentContext of the task
 * Returns:
 *	 - semaphoreId
 * **************************************************************** */

RTDS_SemaphoreId RTDS_CountingSemaphoreCreate(char *name, int semaphoreNumber, unsigned char initialCount, RTDS_GlobalProcessInfo  *RTDS_currentContext)
	{
	RTDS_SemaphoreId	newSemaphore;

	newSemaphore = (TX_SEMAPHORE *) RTDS_MALLOC(sizeof(TX_SEMAPHORE));

	if ( tx_semaphore_create(newSemaphore, name, initialCount) != TX_SUCCESS )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_COUNTING_SEMAPHORE_CREATE);
	RTDS_Sem_Info_Insert( semaphoreNumber, newSemaphore, RTDS_counting, RTDS_currentContext );

	return newSemaphore;
	}



/* **************************************************************** *
 *	RTDS_Sem_Info_Insert
 * **************************************************************** *
 * Inserts a semaphoreInfo struct in the semaphore info chained list
 * **************************************************************** *
 * Parameters:
 *	 - semaphore number
 *	 - semaphore id
 *	 - semaphore type
 *	 - RTDS_currentContext currentContext of the task
 * Returns:
 *	 The semaphore id so that the user can use the semaphore id directly
 * **************************************************************** */


RTDS_SemaphoreId RTDS_Sem_Info_Insert(
		int 										semaphoreNumber,
		RTDS_SemaphoreId				semaphoreId,
		enum RTDS_SemaphoreType semaphoreType,
		RTDS_GlobalProcessInfo	*RTDS_currentContext)
	{
	RTDS_GlobalSemaphoreInfo	*semInfo, *newSemInfo;

	/* Allocate and fill in a new semInfo structure */
	newSemInfo = (RTDS_GlobalSemaphoreInfo *)RTDS_MALLOC(sizeof(RTDS_GlobalSemaphoreInfo));
	if ( newSemInfo == NULL )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_SEM_INFO_INSERT_MALLOC);

	newSemInfo->semaphoreType 	= semaphoreType;
	newSemInfo->semaphoreNumber = semaphoreNumber;
	newSemInfo->next						= NULL;

	/* Semaphore creation */
	newSemInfo->semaphoreId = semaphoreId;
	if ( newSemInfo->semaphoreId == NULL )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_SEM_INFO_INSERT);

	/* Add the semaphore information to the chained list pointed by the RTDS_globalSemaphoreInfo global variable */
	RTDS_CRITICAL_SECTION_START;
	if ( RTDS_globalSemaphoreInfo == NULL )
		RTDS_globalSemaphoreInfo = newSemInfo;
	else
		{
		/* Let's get to the end of the list */
		for (semInfo = RTDS_globalSemaphoreInfo ; semInfo->next != NULL ; semInfo = semInfo->next) ;
		semInfo->next = newSemInfo;
		}
	RTDS_CRITICAL_SECTION_STOP;

	#ifdef RTDS_SIMULATOR
		{
		ULONG 								current_value;
		RTDS_SemaphoreStatus	tmpStatus;
		CHAR *name;
		TX_THREAD *first_suspended;
		ULONG suspended_count;
		TX_SEMAPHORE *next_semaphore;
		TX_THREAD *owner;
		TX_MUTEX *next_mutex;

		if ( newSemInfo->semaphoreType == RTDS_mutex )
			{
			tmpStatus = tx_mutex_info_get((RTDS_MutexId)(
				newSemInfo->semaphoreId),
				&name,
				&current_value,
				&owner,
				&first_suspended,
				&suspended_count,
				&next_mutex
				);
			/* current_value is actually the ownership count of the mutex. */
			if (current_value == 0)
				current_value = 1;
			else
				current_value = 0;
			}
		else
			/* current value is the current semaphore�s count. */
			tmpStatus = tx_semaphore_info_get(
				newSemInfo->semaphoreId,
				&name,
				&current_value,
				&first_suspended,
				&suspended_count,
				&next_semaphore
				);
		if ( tmpStatus != RTDS_OK )
			current_value = (ULONG) -1; /* Aouch: -1 on an unsigned long... */


		RTDS_SIMULATOR_TRACE(RTDS_semaphoreCreated, semaphoreId, current_value, RTDS_currentContext);
		}
	#endif

	return semaphoreId;
	}





/* **************************************************************** *
 *	RTDS_Sem_Delete
 * **************************************************************** *
 * Kills a semaphore and delete its info from the semaphore
 * information chained list and free the related memory
 * **************************************************************** *
 * Parameters:
 *		 - the address of the semaphore to find the semaphore info
 *		 block
 *		 - RTDS_currentContext currentContext of the task
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_Sem_Delete( RTDS_SemaphoreId semaphoreId, RTDS_GlobalProcessInfo	*RTDS_currentContext)
	{
	RTDS_GlobalSemaphoreInfo *semInfo, *semInfoPrev;

	RTDS_SIMULATOR_TRACE(RTDS_semaphoreDeleted, semaphoreId, 0, RTDS_currentContext);

	/* Remove the semaphore information from the chained list */
	semInfoPrev = NULL;
	RTDS_CRITICAL_SECTION_START;
	for (semInfo = RTDS_globalSemaphoreInfo ; semInfo != NULL ; semInfo = semInfo->next)
		{
		if ( semInfo->semaphoreId == semaphoreId) /* semaphore found */
			{
			if (semInfoPrev == NULL)
				RTDS_globalSemaphoreInfo = semInfo->next;
			else
				semInfoPrev->next = semInfo->next;

			/* mutex has different primitives */
			if ( semInfo->semaphoreType == RTDS_mutex )
				tx_mutex_delete( (RTDS_MutexId)(semInfo->semaphoreId) );
			else
				tx_semaphore_delete(semInfo->semaphoreId);

			RTDS_FREE(semInfo);
			break;
			}
		semInfoPrev = semInfo;
		}

	RTDS_CRITICAL_SECTION_STOP;

	}




/* **************************************************************** *
 *	RTDS_SemaphoreIdTake
 * **************************************************************** *
 * Attempt to take a semaphore from its id
 * **************************************************************** *
 * Parameters:
 *	 - the id of the semaphore
 *	 - timeout value
 *	 - RTDS_currentContext currentContext of the task
 * Returns:
 *	 - status
 * **************************************************************** */

RTDS_SemaphoreStatus RTDS_SemaphoreIdTake(
	RTDS_SemaphoreId					semaphoreId,
	RTDS_SemaphoreTimeout 		TIME_OUT,
	RTDS_GlobalProcessInfo		*RTDS_currentContext )
	{
	RTDS_SemaphoreStatus			status;
	RTDS_GlobalSemaphoreInfo	*semInfo;

	RTDS_SIMULATOR_TRACE(RTDS_semTakeAttempt, semaphoreId, TIME_OUT, RTDS_currentContext);

	for (semInfo = RTDS_globalSemaphoreInfo ; semInfo != NULL ; semInfo = semInfo->next)
		{
		if ( semInfo->semaphoreId == semaphoreId) /* semaphore found */
			break;
		}

		/* mutex has different primitives */
		if ( semInfo->semaphoreType == RTDS_mutex )
			status = tx_mutex_get( (RTDS_MutexId)(semInfo->semaphoreId), TIME_OUT );
		else
			status = tx_semaphore_get( semInfo->semaphoreId, TIME_OUT );

	if ( status == RTDS_OK )
		{
		#ifdef RTDS_SIMULATOR
		ULONG 								current_value;
		RTDS_SemaphoreStatus	tmpStatus;
		CHAR *name;
		TX_THREAD *first_suspended;
		ULONG suspended_count;
		TX_SEMAPHORE *next_semaphore;
		TX_THREAD *owner;
		TX_MUTEX *next_mutex;

		if ( semInfo->semaphoreType == RTDS_mutex )
			{
			tmpStatus = tx_mutex_info_get((RTDS_MutexId)(
				semInfo->semaphoreId),
				&name,
				&current_value,
				&owner,
				&first_suspended,
				&suspended_count,
				&next_mutex
				);
			/* current_value is actually the ownership count of the mutex. */
			if (current_value == 0)
				current_value = 1;
			else
				current_value = 0;
			}
		else
			/* current value is the current semaphore�s count. */
			tmpStatus = tx_semaphore_info_get(
				semInfo->semaphoreId,
				&name,
				&current_value,
				&first_suspended,
				&suspended_count,
				&next_semaphore
				);
		if ( tmpStatus != RTDS_OK )
			current_value = (ULONG) -1; /* Aouch: -1 on an unsigned long... */

		RTDS_SIMULATOR_TRACE(RTDS_semTakeSucceded, semaphoreId, current_value, RTDS_currentContext);
		#endif
		}
	else
		{
		status = RTDS_ERROR;
		RTDS_SIMULATOR_TRACE(RTDS_semTakeTimedOut, semaphoreId, NULL, RTDS_currentContext);
		}
	return status;
	}




/* **************************************************************** *
 *	RTDS_GetSemaphoreId
 * **************************************************************** *
 * Gets the id of a semaphore from its number (name)
 * **************************************************************** *
 * Parameters:
 *	 - semaphore number representing its name
 * Returns:
 *	 - the id of the semaphore
 * **************************************************************** */

RTDS_SemaphoreId RTDS_GetSemaphoreId( int semaphoreNumber )
	{
	RTDS_GlobalSemaphoreInfo	*semInfo;
	RTDS_SemaphoreId			foundSemaphoreId;

	foundSemaphoreId = NULL;
	RTDS_CRITICAL_SECTION_START;
	for (semInfo = RTDS_globalSemaphoreInfo ; semInfo != NULL ; semInfo = semInfo->next)
		{
		if ( semInfo->semaphoreNumber == semaphoreNumber ) /* semaphore found */
			{
			foundSemaphoreId = semInfo->semaphoreId;
			break;
			}
		}
	RTDS_CRITICAL_SECTION_STOP;

	if (foundSemaphoreId==NULL)
		RTDS_SYSTEM_ERROR(RTDS_ERROR_GET_SEMAPHORE_ID);

	return foundSemaphoreId;	/* Might cause an RTOS exception if NULL*/
	}




/* **************************************************************** *
 *	RTDS_GetSemaphoreInfo
 * **************************************************************** *
 * Gets the information structure of a semaphore from its id
 * **************************************************************** *
 * Parameters:
 *	 - the id of the semaphore
 * Returns:
 *	 - the info structure of the semaphore
 * **************************************************************** */

RTDS_GlobalSemaphoreInfo *RTDS_GetSemaphoreInfo( RTDS_SemaphoreId semaphoreId )
	{
	RTDS_GlobalSemaphoreInfo				*semInfo;

	for (semInfo = RTDS_globalSemaphoreInfo ; semInfo != NULL ; semInfo = semInfo->next)
		{
		if ( semInfo->semaphoreId == semaphoreId) /* semaphore found */
			return semInfo;
		}

	/* Should never get here */
	RTDS_SYSTEM_ERROR(RTDS_ERROR_GET_SEMAPHORE_INFO);
	return NULL;
	}



/* **************************************************************** *
 *	RTDS_TransitionCleanUp
 * **************************************************************** *
 * Called at the end of transitions:
 * - frees message buffer if valid
 * - re-organize save queue if state has changed
 * **************************************************************** *
 * Parameters:
 *	 None
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_TransitionCleanUp(RTDS_GlobalProcessInfo *RTDS_currentContext, int RTDS_sdlStatePrev)
	{
	/* Free message buffer if valid */
	if (RTDS_currentContext->currentMessage != NULL )
		{
    /* Free the message parameter structure if any */
    if (RTDS_currentContext->currentMessage->pData != NULL){
      RTDS_FREE(RTDS_currentContext->currentMessage->pData);
      }

		#ifdef RTDS_SIMULATOR
			/* Release the message unique id back to the pool */
			RTDS_ReleaseMessageUniqueId(RTDS_currentContext->currentMessage->messageUniqueId);
		#endif

		/* Free memory */
		RTDS_FREE(RTDS_currentContext->currentMessage);
		RTDS_currentContext->currentMessage = NULL;
		} /* End of if ( RTDS_currentContext->currentMessage != NULL ) */

	/* If SDL state has changed and messages have been saved: reorganise the save queue */
	if ( (RTDS_currentContext->sdlState != RTDS_sdlStatePrev) && (RTDS_currentContext->writeSaveQueue != NULL) )
		{
		RTDS_MessageHeader	*message;
		/* Let's get to the end of the save queue */
		for (message=RTDS_currentContext->writeSaveQueue ; message->next != NULL ; message = message->next);
		message->next = RTDS_currentContext->readSaveQueue;
		RTDS_currentContext->readSaveQueue = RTDS_currentContext->writeSaveQueue;
		RTDS_currentContext->writeSaveQueue = NULL;
		}

	return;
	}



/* **************************************************************** *
 *	RTDS_GetSystemTime
 * **************************************************************** *
 * As its name states...
 * **************************************************************** *
 * Parameters:
 *	 None
 * Returns:
 *	 System tick count value
 * **************************************************************** */

unsigned long RTDS_GetSystemTime(void)
	{
	return tx_time_get();
	}




/* **************************************************************** *
 * **************************************************************** *
 * **************************************************************** *
 * THE CODE BELOW IS ONLY USED TO DEBUG WITH RTDS SDL-RT DEBUGGER
 * **************************************************************** *
 * **************************************************************** *
 * **************************************************************** */

#ifdef RTDS_SIMULATOR

unsigned char 				*RTDS_globalMessageUniqueIdPool=NULL;
RTDS_GlobalTraceInfo	RTDS_globalTraceEntry={RTDS_systemError,NULL,0};
char									*RTDS_globalPrintableParameters=NULL;

/* **************************************************************** *
 *	RTDS_GetMessageUniqueId
 * **************************************************************** *
 * Gets a message unique id for the simulator
 * **************************************************************** *
 * Parameters:
 *	 None
 * Returns:
 *	 - the message unique id (minimum is 1)
 * **************************************************************** */

unsigned long RTDS_GetMessageUniqueId(void)
	{
	unsigned char *index;
	long		uniqueByteId;
	long		uniqueBitId;

	index=RTDS_globalMessageUniqueIdPool;

	RTDS_CRITICAL_SECTION_START;
	for (uniqueByteId=0;uniqueByteId < RTDS_MESSAGE_UNIQUE_ID_POOL_SIZE;uniqueByteId++)
	{
	if (*index != 0xFF)
		{
		for (uniqueBitId=0;uniqueBitId<8;uniqueBitId++)
		{
		if ( ( (1<<uniqueBitId) & *index) ==0 )
			{
			*index = *index | (1<<uniqueBitId) ;
			RTDS_CRITICAL_SECTION_STOP;
			return 8*uniqueByteId+uniqueBitId+1;
			}
		}
		}
	index++;
	}
	RTDS_CRITICAL_SECTION_STOP;

	/* All bits are set... No more message unique id */
	RTDS_SYSTEM_ERROR(RTDS_ERROR_NO_MORE_MSG_UNIQUE_ID);
	return 0;
	}




/* **************************************************************** *
 *	RTDS_ReleaseMessageUniqueId
 * **************************************************************** *
 * Make a message unique id available from the pool
 * **************************************************************** *
 * Parameters:
 *	 message unique id
 * Returns:
 *	 nothing
 * **************************************************************** */

void RTDS_ReleaseMessageUniqueId(unsigned long messageUniqueId)
	{
	unsigned char *index;

	if (messageUniqueId == 0) /* probably a timer */
		return;

	messageUniqueId -= 1;
	index = RTDS_globalMessageUniqueIdPool;
	index += (unsigned char)(messageUniqueId/8);
	RTDS_CRITICAL_SECTION_START;
	(*index) = (*index) & ~(1<<messageUniqueId%8);
	RTDS_CRITICAL_SECTION_STOP;
	}



/* **************************************************************** *
 *	RTDS_DummyTraceFunction
 * **************************************************************** *
 * As its name states... The simulator sets a breakpoint on this
 * function and reads the RTDS_globalTraceEntry variable to see
 * what happened
 * **************************************************************** *
 * Parameters:
 *	 None
 * Returns:
 *	 nothing
 * **************************************************************** */

void RTDS_DummyTraceFunction(void)
	{
	/* The print below can be activated with the following options in the generation profile:
	 * -DRTDS_FORMAT_TRACE -I${RTDS_HOME}\share\ccg\trace\DynamicTrace
	 * And with $(RTDS_HOME)/trace/formattrace/RTDS_FormatTrace.c added in the project
	 * The printout can be sent as is to the MSC tracer.
	 */
	#ifdef RTDS_FORMAT_TRACE
		char *formatedTrace;

		RTDS_FormatTrace(
			(unsigned long)RTDS_GetSystemTime(),	/* System time */
			RTDS_globalTraceEntry,								/* Trace info */
			RTDS_DTRACE_ACK_NOWAIT, 							/* No acknowledgment */
			&formatedTrace);											/* Buffer will be allocated by RTDS_FormatDynamicTrace */
		printf(formatedTrace);
		RTDS_FREE((void *)formatedTrace);
	#endif
	}

#endif
