#include <stdio.h>

#include <RTDS_TTCN.h>
#include "RTDS_MACRO.h"
#include "RTDS_TTCN_MACRO.h"
#include "RTDS_OS.h"
#include "RTDS_String.h"
#include <RTDS_BasicTypes.h>
#include <RTDS_Common.h>
#include <tci/tci.h>

/*
** The following are declarations that are only valid in a TTCN-only non-debug context. If SDL
** is generated with the TTCN, or if generation is done for debug, these declarations will be
** in RTDS_Start.
*/
#if !defined(RTDS_TTCN_WITH_SDL) && !defined(RTDS_SIMULATOR)
/* Pointers to global list of information on processes and semaphores */
RTDS_GlobalProcessInfo    * RTDS_globalProcessInfo = NULL;
RTDS_GlobalSemaphoreInfo  * RTDS_globalSemaphoreInfo = NULL;

/* Various declarations for RTDS feature and critical sections */
RTDS_COVERAGE_DECL;
RTDS_SYS_TIME_DECL;
RTDS_START_SYNCHRO_DECL;
RTDS_CRITICAL_SECTION_DECL;
RTDS_SOCKET_ACCESS_DECL;
RTDS_CRITICAL_TRACE_SECTION_DECL;
RTDS_DTRACE_ACKNOWLEDGE_DECL;
#endif


/*
** RTDS_TTCN_Init:
** ---------------
** Performs all necessary initializations in a TTCN system. Must be called
** first in any generated executable.
*/
void RTDS_TTCN_Init(void)
  {
  /* Perform necessary initializations for RTDS features and critical sections if they are not done at SDL level */
#if !defined(RTDS_TTCN_WITH_SDL) && !defined(RTDS_SIMULATOR)
  RTDS_SYS_TIME_INIT;
  RTDS_COVERAGE_INIT;
  RTDS_DTRACE_ACKNOWLEDGE_INIT;
  RTDS_SOCKET_ACCESS_INIT;
  RTDS_CRITICAL_SECTION_INIT;
  RTDS_START_SYNCHRO_INIT;
  RTDS_CRITICAL_TRACE_SECTION_INIT;
#endif
  /* TTCN-specific initializations */
  /* (to be continued...) */
  }

/*
** RTDS_TTCN_Init:
** ---------------
** Performs all necessary cleanup in a TTCN system. Must be called
** before exiting.
*/
void RTDS_TTCN_Postamble(void)
  {
  /* Perform necessary cleanup for RTDS features and critical sections if they are not done at SDL level */
#if !defined(RTDS_TTCN_WITH_SDL) && !defined(RTDS_SIMULATOR)
  RTDS_DTRACE_ACKNOWLEDGE_POSTAMBLE;
  RTDS_SOCKET_ACCESS_POSTAMBLE;
  RTDS_CRITICAL_SECTION_POSTAMBLE;
  RTDS_START_SYNCHRO_POSTAMBLE;
  RTDS_CRITICAL_TRACE_SECTION_POSTAMBLE;
#endif
  }  
  
/*
** RTDS_Register_Template_in_Pool:
** -------------------------------
** Add template to template pool and return it.
*/
RTDS_TTCN_Template * RTDS_TTCN_RegisterTemplatePool( RTDS_TTCN_Template * templateToRegister)
  {
  RTDS_TTCN_Template_Pool * newPoolElement = (RTDS_TTCN_Template_Pool*)RTDS_MALLOC(sizeof(RTDS_TTCN_Template_Pool));
  newPoolElement->templateID = templateToRegister;
  newPoolElement->next = RTDS_TTCN_globalTemplatePool;
  RTDS_TTCN_globalTemplatePool = newPoolElement;
    
  RTDS_TTCN_Template_Pool * RTDS_TTCN_globalTemplatePool_tmp = RTDS_TTCN_globalTemplatePool;
  while (RTDS_TTCN_globalTemplatePool_tmp != NULL)
    {
    RTDS_TTCN_globalTemplatePool_tmp = RTDS_TTCN_globalTemplatePool_tmp->next;
    }
    
  return templateToRegister;
  }

/*
** RTDS_TTCN_UpdateFirstMessage:
** -------------------------------
** Update first message of every port in component.
*/    
void RTDS_TTCN_UpdateFirstMessage ( RTDS_TTCN_GlobalComponentInfo * currentComponentInfo )
  {
  RTDS_GlobalProcessInfo        * RTDS_currentContext;
  RTDS_TTCN_PortMappingInfo     * portMapping;

  RTDS_CRITICAL_SECTION_START;
  RTDS_currentContext = currentComponentInfo->RTDS_currentContext;
  for (portMapping = currentComponentInfo->portMappingInfo; portMapping != NULL; portMapping = portMapping->next)
    {
    if (portMapping->currentMessage == NULL)
      {
      if (portMapping->queueControlBlock->queue != NULL)
        {
        RTDS_CHECKED_PTR_MALLOC(portMapping->currentMessage, 1, RTDS_MessageHeader);
        RTDS_MSG_QUEUE_ID_READ(portMapping->queueControlBlock, portMapping->currentMessage);
        }
      }
    }
  RTDS_CRITICAL_SECTION_STOP;
  }
    
/*
** RTDs_TTCN_UpdateTimerLists:
** -------------------------------
** Update timers lists for timeout.
*/ 
void RTDS_TTCN_UpdateTimerLists ( RTDS_TTCN_GlobalComponentInfo * RTDS_TTCN_currentComponent )
  {
  RTDS_TTCN_TimerInfoList * timerInfoList;
  RTDS_TIMER timerInfo;
  for (timerInfoList = RTDS_TTCN_currentComponent->timerInfoList ; timerInfoList!=NULL ; timerInfoList = timerInfoList->next)
    {
    timerInfo = timerInfoList->timerInfo;
    if (timerInfo->timerStatus == runningT)
      {
      if (timerInfo->timeoutValue < RTDS_GET_SYSTEM_TIME)
        {
        timerInfo->timerStatus = expiredT;
        }
      }
    }
  }
    
/*
** RTDS_SetStatus:
** ---------------
** Set component status.
*/
void RTDS_TTCN_SetStatus(RTDS_SdlInstanceId * componentToChange, ComponentStatusType status)
  {
  RTDS_TTCN_GlobalComponentInfo * componentInfo;

  /* Look for component to change in global component list */
  for (componentInfo = RTDS_TTCN_globalComponentInfo; componentInfo != NULL; componentInfo = componentInfo->next)
    {
    /* If current component is component to change, set its status */
    if (componentInfo->RTDS_currentContext->mySdlInstanceId == componentToChange)
      {
      componentInfo->componentStatus = status;
      return;
      }
    }
  /* If component not found, error */
  RTDS_SYSTEM_ERROR(RTDS_ERROR_COMPONENT_NOT_FOUND);
  }

/*
** RTDS_CheckAlive:
** ----------------
** Check if a component is alive.
*/
int RTDS_TTCN_CheckAlive(RTDS_SdlInstanceId * componentToCheck)
  {
  RTDS_TTCN_GlobalComponentInfo * componentInfo;

  /* Look for component to check in global component list */
  for (componentInfo = RTDS_TTCN_globalComponentInfo; componentInfo != NULL; componentInfo = componentInfo->next)
    {
    /* If current component is component to check */
    if (componentInfo->RTDS_currentContext->mySdlInstanceId == componentToCheck)
      {
      /* If the verdict of this component is inactive, running or stopped, component is alive */
      if ((componentInfo->componentStatus == inactiveC) || (componentInfo->componentStatus == runningC) || (componentInfo->componentStatus == stoppedC))
        return 1;
      }
    }
    /* If component not found, or verdict is not good, component is inactive */
    return 0;
  }

/*
** RTDS_CheckRunning:
** ------------------
** Check if a component is running.
*/    
int RTDS_TTCN_CheckRunning(RTDS_SdlInstanceId * componentToCheck)
  {
  RTDS_TTCN_GlobalComponentInfo * componentInfo;
  int isRunning = 0;
  
  RTDS_CRITICAL_SECTION_START;
  /* Look for component to check in global component list */
  for (componentInfo = RTDS_TTCN_globalComponentInfo; componentInfo != NULL; componentInfo = componentInfo->next)
    {
    /* If current component is component to check */
    if (componentInfo->RTDS_currentContext->mySdlInstanceId == componentToCheck)
      {
      /* Checking if component is running via its status */
      if (componentInfo->componentStatus == runningC)
        {
        isRunning = 1;
        break;
        }
      }
    }
  RTDS_CRITICAL_SECTION_STOP;
  
  /* If component not found, or status is not good, component is inactive */
  return isRunning;
  }

/*
** RTDS_CheckStopped:
** ------------------
** Check if a component is stopped.
*/   
int RTDS_TTCN_CheckStopped(RTDS_SdlInstanceId * componentToCheck)
  {
  RTDS_TTCN_GlobalComponentInfo * componentInfo;
  int isStopped = 0;
  
  RTDS_CRITICAL_SECTION_START;
  /* Look for component to check in global component list */
  for (componentInfo = RTDS_TTCN_globalComponentInfo; componentInfo != NULL; componentInfo = componentInfo->next)
    {
    /* If current component is component to check, check if it's stopped via its status */
	  if (componentInfo->RTDS_currentContext->mySdlInstanceId == componentToCheck)
      {
	    // return 1 if component is killed or stopped.
	    isStopped = ((componentInfo->componentStatus == stoppedC) || (componentInfo->componentStatus == killedC));
	    break;
	    }
    }
  RTDS_CRITICAL_SECTION_STOP;
  
  /* If component not found, assume it is stopped */
	return isStopped;
  }

/*
** RTDS_CheckKilled:
** -----------------
** Check if a component is killed.
*/  
int RTDS_TTCN_CheckKilled(RTDS_SdlInstanceId * componentToCheck)
  {
  RTDS_TTCN_GlobalComponentInfo * componentInfo;
  int isKilled = 0;
  
  RTDS_CRITICAL_SECTION_START;
  /* Look for component to check in global component list */
  for (componentInfo = RTDS_TTCN_globalComponentInfo; componentInfo != NULL; componentInfo = componentInfo->next)
    {
    /* If current component is component to check */
    if (componentInfo->RTDS_currentContext->mySdlInstanceId == componentToCheck)
      {
      /* Checking if component is killed via its status */
      if (componentInfo->componentStatus == killedC)
        {
        isKilled = 1;
        break;
        }
      }
    }
  RTDS_CRITICAL_SECTION_STOP;
  
  /* If component not found, or status is not good, assume it is stopped */
  return isKilled;
  }

/*
** RTDS_GetVerdict:
** ----------------
** Return the verdict for a component.
*/  
verdicttype RTDS_TTCN_GetVerdict(RTDS_SdlInstanceId * component)
  {
  RTDS_TTCN_GlobalComponentInfo * componentInfo;

  /* Look for component in global component list */
  for (componentInfo = RTDS_TTCN_globalComponentInfo; componentInfo != NULL; componentInfo = componentInfo->next)
    {
    /* If component found, return its verdict */
    if (componentInfo->RTDS_currentContext->mySdlInstanceId == component)
      return componentInfo->componentVerdict;   
    }
  /* If component not found, error */
  RTDS_SYSTEM_ERROR(RTDS_ERROR_COMPONENT_NOT_FOUND);
  }
    

/*
** RTDS_TTCN_GetFinalVerdict:
** ----------------
** Return the final verdict of a testcase.
*/  
verdicttype RTDS_TTCN_GetFinalVerdict()
  {
  verdicttype finalVerdict = TCI_VERDICT_NONE;
  /* For every components is component list */
  for (RTDS_TTCN_GlobalComponentInfo * RTDS_tmpglobalComponentInfo = RTDS_TTCN_globalComponentInfo; RTDS_tmpglobalComponentInfo!=NULL;RTDS_tmpglobalComponentInfo=  RTDS_tmpglobalComponentInfo->next)
    {
    /* If local verdict is worth, global verdict is local verdict */
    if (finalVerdict<RTDS_tmpglobalComponentInfo->componentVerdict)
      {
      finalVerdict=RTDS_tmpglobalComponentInfo->componentVerdict;
      }
    }
  return finalVerdict;
  }

/*
** RTDS_TTCN_getcurrentTestcase:
** ----------------
** get the name of the current running testcase.
*/  
char * RTDS_TTCN_getcurrentTestcase()
  {
  /* For every components is component list */
  for (RTDS_TTCN_GlobalComponentInfo * tmpGlobalComponentInfo = RTDS_TTCN_globalComponentInfo; tmpGlobalComponentInfo!=NULL;tmpGlobalComponentInfo=tmpGlobalComponentInfo->next)
    {
    if (tmpGlobalComponentInfo->componentKindType == TCI_MTC_COMP)
      {
	  return tmpGlobalComponentInfo->testcaseName;
      }
    }
  return "";
  }
    
/*
** RTDS_TTCN_MsgQueueCheck:
** ------------------------
** Check if there is any message in the queue associated with a port in a given component.
** If there is, the first message is put in the 'message' parameter, which must be allocated
** by the caller, and the function returns true. If there isn't, the function returns false.
*/     
int RTDS_TTCN_MsgQueueCheck(char * portName, RTDS_MessageHeader * message, RTDS_TTCN_GlobalComponentInfo * currentComponent)
  {
  RTDS_TTCN_PortMappingInfo * portMapping;

  /* Look for port in the list associated to the current component */
  for (portMapping = currentComponent->portMappingInfo; portMapping != NULL; portMapping = portMapping->next)
    {
    /* If port is the one to check */
    if (strcmp(portMapping->portId->portName, portName) == 0)
      {
      /* If queue is empty, over */
      if ( portMapping->queueControlBlock->queue == NULL )
        {
        return 0;
        }
      /* If queue is not empty */
      else
        {
        /* Copy first message in the queue */
        RTDS_SemTake(portMapping->queueControlBlock->chainedListSemId, RTDS_SEMAPHORE_TIME_OUT_FOREVER);
        memcpy(message, portMapping->queueControlBlock->queue, sizeof(RTDS_MessageHeader));
        RTDS_SemGive(portMapping->queueControlBlock->chainedListSemId);
        /* Over */
        return 1;
        }
      }
    }
  /* If port not found, assume no message found */
  return 0;
  }

/*
** RTDS_CheckAnyTimerRunning:
** --------------------------
** Check if a least one timer is running in a given component.
*/     
int RTDS_TTCN_CheckAnyTimerRunning (RTDS_TTCN_GlobalComponentInfo * currentComponent)
  {
  RTDS_TTCN_TimerInfoList * timerInfo;

  /* Browse timer list in component */
  for (timerInfo = currentComponent->timerInfoList; timerInfo !=NULL; timerInfo = timerInfo->next)
    {
    /* If timer is running, we have one */
    if (timerInfo->timerInfo->timerStatus == runningT)
      return 1;
    }
  /* If we get there, no timer found or none is running */
  return 0;
  }

/*
** RTDS_TTCN_CheckTimeout:
** ----------------------
** Check if timer is timeout in a given context.
*/     
int RTDS_TTCN_CheckTimeout(RTDS_TIMER timerToWaitFor)
  {
  if (timerToWaitFor->timerStatus == expiredT)
    {
    timerToWaitFor->timerStatus = inactiveT;
    return 0;
    }
  else
    {
    return 1;
    }
  }

/*
** RTDS_TTCN_TimerCreate:
** ----------------------
** Declare new timer with all necessary data.
*/  
RTDS_TIMER RTDS_TTCN_TimerCreate (long timerID, int timerNumber, RTDS_String timerName, TriTimerDuration duration)
  {
  RTDS_TIMER newTimer;
  RTDS_CHECKED_PTR_MALLOC(newTimer,1,RTDS_TTCN_TimerInfo);
  newTimer->timerID = timerID;
  #if defined( RTDS_SIMULATOR )
  newTimer->timerUniqueID = RTDS_GetMessageUniqueId();
  #endif
  newTimer->timerNumber = timerNumber;
  RTDS_StringAssign(newTimer->timerName,timerName);
  newTimer->duration = duration;
  newTimer->timerStatus=inactiveT;
  return newTimer;
  }
  
/*
** FUNCTION RTDS_TTCN_testcaseTerminated:
** --------------------------------------
** Function called when a testcase terminates to trace its final verdict
*/

void RTDS_TTCN_TestcaseTerminated(RTDS_GlobalProcessInfo * RTDS_currentContext, verdicttype verdict)
  {
  static char * verdict_strings[] = {"none", "pass", "inconc", "fail", "error"};
  char message[64];
  
  sprintf(message, "End: %s", verdict_strings[(int)verdict]);
  RTDS_INFORMATION(message);
  }    

  
RTDS_TTCN_ActivateAlstepList * RTDS_TTCN_activateAltstep(char * altstepName, RTDS_TTCN_functionpointer altstepfunction, RTDS_TTCN_GlobalComponentInfo * RTDS_TTCN_currentComponent)
  {
  RTDS_TTCN_ActivateAlstepList * RTDS_altstepToActivate = NULL;
  RTDS_CHECKED_PTR_MALLOC(RTDS_altstepToActivate,1,RTDS_TTCN_ActivateAlstepList);
  RTDS_altstepToActivate->altstepfunction = altstepfunction;
  RTDS_altstepToActivate->altstepName = altstepName;
  RTDS_altstepToActivate->next = RTDS_TTCN_currentComponent->activateAltstepList;
  RTDS_TTCN_currentComponent->activateAltstepList = RTDS_altstepToActivate;
  return RTDS_altstepToActivate;
  }
  
/*
** FUNCTION RTDS_TTCN_int2str:
** --------------------------------------
** Function called for the int2str predefined TTCN-3 function
*/
char * RTDS_TTCN_int2str(int myInt, char * RTDS_TTCN_returnVariable)
  {
  sprintf(RTDS_TTCN_returnVariable,"%d",myInt);
  return RTDS_TTCN_returnVariable;
  }
	
/*
** FUNCTION RTDS_TTCN_int2bit:
** --------------------------------------
** Function called for the int2bit predefined TTCN-3 function
*/
RTDS_BitString * RTDS_TTCN_int2bit(int myInt, int length, RTDS_BitString * RTDS_TTCN_returnVariable)
  {
		
  char message[RTDS_BITSTRING_MAX_SIZE];
  
	memcpy(message,&myInt,sizeof(int));
	message[sizeof(int)] = '\0';
	
	RTDS_StringAssign((char*)RTDS_TTCN_returnVariable->__string,message);
	RTDS_TTCN_returnVariable->__length = length;
	return RTDS_TTCN_returnVariable;
  }

/*
** FUNCTION RTDS_TTCN_int2oct:
** --------------------------------------
** Function called for the int2oct predefined TTCN-3 function
*/
RTDS_OctetString * RTDS_TTCN_int2oct(int myInt, int length, RTDS_OctetString * RTDS_TTCN_returnVariable)
  {
		
  char message[RTDS_BITSTRING_MAX_SIZE];
  
	memcpy(message,&myInt,sizeof(int));
	message[sizeof(int)] = '\0';
	
	RTDS_StringAssign((char*)RTDS_TTCN_returnVariable->__string,message);
	RTDS_TTCN_returnVariable->__length = length;
	return RTDS_TTCN_returnVariable;
  }
	
	
/* Global list of all configurations with module and testcase encoded parameters */
RTDS_TTCN_EncDecConfiguration * RTDS_TTCN_encdecConfigurations = NULL;

/*
** FUNCTION RTDS_TTCN_GetEncDecConfiguration:
** ------------------------------------------
** Returns a configuration with encoded parameters, and can create a new one if it does not exist
*/
RTDS_TTCN_EncDecConfiguration * RTDS_TTCN_GetEncDecConfiguration(char * configurationName, RTDS_BOOLEAN addIfNotFound)
  {
  RTDS_TTCN_EncDecConfiguration * configuration = NULL;
  for (configuration = RTDS_TTCN_encdecConfigurations; configuration != NULL && strcmp(configuration->name, configurationName) != 0; configuration = configuration->next);
  if (configuration == NULL && addIfNotFound)
    {
    configuration = (RTDS_TTCN_EncDecConfiguration *) RTDS_MALLOC(sizeof(RTDS_TTCN_EncDecConfiguration));
    configuration->name = (char *) RTDS_CALLOC(strlen(configurationName) + 1, sizeof(char));
    strcpy(configuration->name, configurationName);
    configuration->modules = NULL;
    configuration->next = RTDS_TTCN_encdecConfigurations;
    RTDS_TTCN_encdecConfigurations = configuration;
    }
  return configuration;
  }

/*
** FUNCTION RTDS_TTCN_GetEncDecModule:
** -----------------------------------
** Returns a module with encoded parameters, and can create a new one if it does not exist
*/
RTDS_TTCN_EncDecModule * RTDS_TTCN_GetEncDecModule(RTDS_TTCN_EncDecConfiguration * configuration, char * moduleName, RTDS_BOOLEAN addIfNotFound)
  {
  if (configuration == NULL)
    {
    return NULL;
    }
  RTDS_TTCN_EncDecModule * module = NULL;
  for (module = configuration->modules; module != NULL && strcmp(module->name, moduleName) != 0; module = module->next);
  if (module == NULL && addIfNotFound)
    {
    module = (RTDS_TTCN_EncDecModule *) RTDS_MALLOC(sizeof(RTDS_TTCN_EncDecModule));
    module->name = (char *) RTDS_CALLOC(strlen(moduleName) + 1, sizeof(char));
    strcpy(module->name, moduleName);
    module->testcases = NULL;
    module->parameters = NULL;
    module->next = configuration->modules;
    configuration->modules = module;
    }
  return module;
  }

/*
** FUNCTION RTDS_TTCN_GetEncDecTestcase:
** -------------------------------------
** Returns a testcase with encoded parameters, and can create a new one if it does not exist
*/
RTDS_TTCN_EncDecTestcase * RTDS_TTCN_GetEncDecTestcase(RTDS_TTCN_EncDecModule * module, char * testcaseName, RTDS_BOOLEAN addIfNotFound)
  {
  if (module == NULL)
    {
    return NULL;
    }
  RTDS_TTCN_EncDecTestcase * testcase = NULL;
  for (testcase = module->testcases; testcase != NULL && strcmp(testcase->name, testcaseName) != 0; testcase = testcase->next);
  if (testcase == NULL && addIfNotFound)
    {
    testcase = (RTDS_TTCN_EncDecTestcase *) RTDS_MALLOC(sizeof(RTDS_TTCN_EncDecTestcase));
    testcase->name = (char *) RTDS_CALLOC(strlen(testcaseName) + 1, sizeof(char));
    strcpy(testcase->name, testcaseName);
    testcase->parameters = NULL;
    testcase->next = module->testcases;
    module->testcases = testcase;
    }
  return testcase;
  }

/*
** FUNCTION RTDS_TTCN_AddEncDecParameter:
** --------------------------------------
** Inserts a new module or testcase encoded parameter in its corresponding list
*/
void RTDS_TTCN_AddEncDecParameter(RTDS_TTCN_EncDecModule * module, RTDS_TTCN_EncDecTestcase * testcase, char * parameterName, char * parameterValue)
  {
  RTDS_TTCN_EncDecParameter * parameter =  (RTDS_TTCN_EncDecParameter *) RTDS_MALLOC(sizeof(RTDS_TTCN_EncDecParameter));
  parameter->name = (char *) RTDS_CALLOC(strlen(parameterName) + 1, sizeof(char));
  strcpy(parameter->name, parameterName);
  int len = strlen(parameterValue) + 1;
  parameter->value = (char *) RTDS_CALLOC(len, sizeof(char));
  int i, j;
  for (i = 0, j = 0; i < len - 1; ++i, ++j)
    {
    if (parameterValue[i] == '\\')
      {
      ++i;
      switch (parameterValue[i])
        {
        case 't':
          parameter->value[j] = '\t';
          break;
        case 'r':
          parameter->value[j] = '\r';
          break;
        case 'n':
          parameter->value[j] = '\n';
          break;
        default:
          parameter->value[j] = parameterValue[i];
        }
      }
    else
      {
      parameter->value[j] = parameterValue[i];
      }
    }
  if (testcase != NULL)
    {
    parameter->next = testcase->parameters;
    testcase->parameters = parameter;
    }
  else if (module != NULL)
    {
    parameter->next = module->parameters;
    module->parameters = parameter;
    }
  else
    {
    RTDS_FREE(parameter->name);
    RTDS_FREE(parameter->value);
    RTDS_FREE(parameter);
    }
  }

/*
** FUNCTION RTDS_TTCN_GetEncDecParameter:
** --------------------------------------
** Returns a module or testcase encoded parameter found on its corresponding list
*/
RTDS_TTCN_EncDecParameter * RTDS_TTCN_GetEncDecParameter(RTDS_TTCN_EncDecModule * module, RTDS_TTCN_EncDecTestcase * testcase, char * parameterName)
  {
  RTDS_TTCN_EncDecParameter * parameter = NULL;
  if (testcase != NULL)
    {
    for (parameter = testcase->parameters; parameter != NULL && strcmp(parameter->name, parameterName) != 0; parameter = parameter->next);
    }
  else if (module != NULL)
    {
    for (parameter = module->parameters; parameter != NULL && strcmp(parameter->name, parameterName) != 0; parameter = parameter->next);
    }
  return parameter;
  }

/*
** FUNCTION RTDS_TTCN_ClearEncDecParameters:
** -----------------------------------------
** Clean-up all module and testcase encoded parameters
*/
void RTDS_TTCN_ClearEncDecParameters()
  {
  RTDS_TTCN_EncDecConfiguration * configuration = RTDS_TTCN_encdecConfigurations;
  while (RTDS_TTCN_encdecConfigurations != NULL)
    {
    RTDS_TTCN_EncDecConfiguration * configuration2Clear = RTDS_TTCN_encdecConfigurations;
    RTDS_TTCN_encdecConfigurations = RTDS_TTCN_encdecConfigurations->next;
    while (configuration2Clear->modules != NULL)
      {
      RTDS_TTCN_EncDecModule * module2Clear = configuration2Clear->modules;
      configuration2Clear->modules = configuration2Clear->modules->next;
      while (module2Clear->parameters != NULL)
        {
        RTDS_TTCN_EncDecParameter * parameter2Clear = module2Clear->parameters;
        module2Clear->parameters = module2Clear->parameters->next;
        RTDS_FREE(parameter2Clear->name);
        RTDS_FREE(parameter2Clear->value);
        RTDS_FREE(parameter2Clear);
        }
      while (module2Clear->testcases != NULL)
        {
        RTDS_TTCN_EncDecTestcase * testcase2Clear = module2Clear->testcases;
        module2Clear->testcases = module2Clear->testcases->next;
        while (testcase2Clear->parameters != NULL)
          {
          RTDS_TTCN_EncDecParameter * parameter2Clear = testcase2Clear->parameters;
          testcase2Clear->parameters = testcase2Clear->parameters->next;
          RTDS_FREE(parameter2Clear->name);
          RTDS_FREE(parameter2Clear->value);
          RTDS_FREE(parameter2Clear);
          }
        RTDS_FREE(testcase2Clear->name);
        RTDS_FREE(testcase2Clear);
        }
      RTDS_FREE(module2Clear->name);
      RTDS_FREE(module2Clear);
      }
    RTDS_FREE(configuration2Clear->name);
    RTDS_FREE(configuration2Clear);
    }
  }  
