#include <stdio.h>
#include <stdlib.h>

#include "tci/tci.h"
#include "tci/tciRequired.h"
#include "RTDS_TTCN_MACRO.h"
#include "RTDS_TTCN.h"

/* Include unistd when using a unix based system or cygwin */
#ifdef __CYGWIN__
#include <unistd.h>
#else
#include <stdlib.h>
#endif


/*
** Declarations of functions provided by the generated code, but that are not
** in the TCI required interface.
*/
extern TciModuleIdListType * RTDS_TTCN_GetAllRootModules();
extern verdicttype RTDS_TTCN_ExecuteTestcase(char *, RTDS_TTCN_EncDecTestcase *);
extern int RTDS_TTCN_GetProcessNumber(TciModuleIdType *);
extern RTDS_BOOLEAN RTDS_TTCN_init_modulepar(RTDS_TTCN_EncDecConfiguration *);
extern void RTDS_TTCN_init_const();

extern char * RTDS_TTCN_defaultRootModule;


#ifdef RTDS_TTCN_SUT_INIT
int RTDS_TTCN_SUT_INIT(void);
#endif

/* Needed for compilation, but is not used (for now) */
#ifndef RTDS_TTCN_WITH_SDL
int globalClientSocketId = 0;
#endif

/* Cleanup macro */
#ifdef RTDS_TTCN_SUT_CLEAN
int RTDS_TTCN_SUT_CLEAN(void);
#define RTDS_TTCN_RETURN(VALUE) \
  do { \
    RTDS_TTCN_SUT_CLEAN(); \
    RTDS_TTCN_Postamble(); \
    return VALUE; \
  } while (0)
#else
#define RTDS_TTCN_RETURN(VALUE) \
  do { \
    RTDS_TTCN_Postamble(); \
    return VALUE; \
  } while (0)
#endif


/* Maximum length for a name */
#define MAX_NAME 256


/*
** STATIC VARIABLES:
** -----------------
*/

static  int errorIndicator = 0;                         /* Set to true if an error occurs */
static  RTDS_SemaphoreId controlPartTerminated = NULL;  /* Binary semaphore to notify end of control part */
static  int globalVerdict = TCI_VERDICT_NONE;           /* Global verdict for all excuted testcases */

/* String representations for verdicts */
static char * verdict_string[] = { "none", "pass", "inconc", "fail", "error" };

/* Type for linked list of strings, allowing to sort list of modules or testcases */
typedef struct _StringLinkedList
  {
  char                      * string;
  int                         chosenIndex;
  struct _StringLinkedList  * next;
  } StringLinkedList;

/*
** FUNCTION sortedInsert:
** ----------------------
** Utility function to insert a string in a sorted string linked list.
*/

static StringLinkedList * sortedInsert(StringLinkedList * list, char * string)
  {
  StringLinkedList  * element;
  StringLinkedList  * previous;
  StringLinkedList  * current;
  
  element = (StringLinkedList*) RTDS_MALLOC(sizeof(StringLinkedList));
  element->string = string;
  element->chosenIndex = 0;
  element->next = NULL;
  if (list == NULL)
    {
    return element;
    }
  previous = NULL;
  current = list;
  while (current != NULL)
    {
    if (strcmp(element->string, current->string) <= 0)
      {
      element->next = current;
      if (previous == NULL)
        {
        list = element;
        }
      else
        {
        previous->next = element;
        }
      break;
      }
    if (current->next == NULL)
      {
      current->next = element;
      break;
      }
    previous = current;
    current = current->next;
    }

  return list;
  }


/*
** FUNCTION sortedInsert:
** ----------------------
** Utility function to free a whole string linked list.
*/

static void freeStringLinkedList(StringLinkedList * list)
  {
  StringLinkedList  * next;
  while ( list != NULL )
    {
    next = list->next;
    RTDS_FREE(list);
    list = next;
    }
  }


/*
** FUNCTION tciError:
** -----------------------
** Part of the TCI provided interface. Called when an error occurs in the test.
*/

void tciError(char* message)
  {
  printf("!! Error!: %s\n" , message);
  }


/*
** FUNCTION tciControlTerminated:
** ------------------------------
** Part of the TCI provided interface. Called when the execution of a control part ends.
*/

void tciControlTerminated()
  {
  if (controlPartTerminated != NULL)
    {
    RTDS_SemGive(controlPartTerminated);
    }
  }


/*
** FUNCTION tciTestCaseStarted:
** ----------------------------
** Part of the TCI provided interface. Called when the execution of a testcase starts.
*/

void tciTestCaseStarted(TciTestCaseIdType * testCaseId, TciParameterListType * parameterList, double timer)
  {
  printf("-> Testcase %s started\n" , testCaseId->baseName);
  }


/*
** FUNCTION tciTestCaseTerminated:
** -------------------------------
** Part of the TCI provided interface. Called when the execution of a testcase ends.
*/

void tciTestCaseTerminated(VerdictValue verdict, TciParameterListType* parameterlist)
  {
  printf("<- Verdict is: %s\n", verdict_string[verdict]);
  if (globalVerdict < verdict)
    {
    globalVerdict = verdict;
    }
  }


/*
** FUNCTION executeControlPart:
** ----------------------------
** Executes the control part in the current root module if any.
*/

static int executeControlPart(void)
  {
  TriComponentId  * startedControlpart;
  verdicttype       finalVerdict;
  
  printf("=> Starting control part...\n");
  controlPartTerminated = RTDS_SemBCreate(0);
  globalVerdict = TCI_VERDICT_NONE;
  startedControlpart = tciStartControl();
  if (startedControlpart == NULL)
    {
    printf("== Error while starting control part!\n");
    RTDS_SemDelete(controlPartTerminated);
    return TCI_VERDICT_ERROR;
    }
  if (RTDS_SemTake(controlPartTerminated, RTDS_SEMAPHORE_TIME_OUT_FOREVER) != RTDS_OK)
    {
    printf("== Error while waiting for verdict!\n");
    RTDS_SemDelete(controlPartTerminated);
    RTDS_FREE(startedControlpart);
    return TCI_VERDICT_ERROR;
    }
  printf("<= Control part terminated\n");
  RTDS_SemDelete(controlPartTerminated);
  RTDS_FREE(startedControlpart);
  return (int)globalVerdict;
  }


/*
** FUNCTION executeTestcase:
** -------------------------
** Executes a testcase identified by its name in the current root module.
*/

static int executeTestcase(char * testcaseName, RTDS_TTCN_EncDecTestcase * encdecTestcase)
  {
  return (int) RTDS_TTCN_ExecuteTestcase(testcaseName, encdecTestcase);
  }


/*
** FUNCTION main:
** ==============
** If called with a command in its arguments, executes this command in the default module,
** which is the one the code generation has been run on. If called without parameters,
** allows to run control parts or testcases in all known modules interactively.
*/

int main(int argc, char *argv[])
	{
  QualifiedName           currentModuleId;              /* TCI identifier for the current module */
  char                    currentModuleName[MAX_NAME];  /* Name for the current module */
	TciModuleIdListType   * moduleList;                   /* List of available modules */
	int                     moduleIndex;                  /* Index of a module in the previous list */
  StringLinkedList      * sortedNames;                  /* Sorted linked list of strings, i.e module or testcase names */
  StringLinkedList      * sortedNamesElement;           /* Element in the previous list */
  QualifiedName           tempModuleId;                 /* Temporary TCI identifier for tests */
  unsigned char           hasControl;                   /* Indicates whether a module has a control part or not */
  unsigned int            nbTestcases;                  /* Number of testcases in a module */
  TciTestCaseIdListType * testcasesList;                /* List of testcases in a module */
  int                     testcaseIndex;                /* Index of a testcase in the previous list */
  int                     finalVerdict;                 /* Final verdict in testcase or control part execution */
  int                     configurationIndex;           /* Index of a configuration in the list */
  char                    chosenOption[10];             /* Chosen option in interactive mode */
  int                     chosenIndex;                  /* Chosen index of configuration, module, or testcase in interactive mode */

  /* Initializations for TTCN-3 */
  RTDS_TTCN_Init();
  RTDS_TTCN_init_const();
  /* Start SUT if any */
#ifdef RTDS_TTCN_SUT_INIT
  RTDS_TTCN_SUT_INIT();
#endif
  
  /* Command line options */
  char                  * configFileName      = NULL;   /* File (name or path) with configurations of TTCN-3 parameters */
  char                  * configName          = NULL;   /* Configuration name to read TTCN-3 parameters from */
  unsigned char           haveStart           = 0;      /* If 1, execute control part or given testcase automatically, otherwise enter interactive mode */
  char                  * startTestcaseName   = NULL;   /* Testcase to execute automatically */
  /* Parse options */
  int argi = 1, argerr = 0;
  while (argi < argc && argerr == 0)
    {
    /* Configuration file "-f" or "--config-file" */
    if (strncmp(argv[argi], "-f", 2) == 0)
      {
      if (strlen(argv[argi]) > 2)
        configFileName = &argv[argi][2];
      else
        {
        ++argi;
        if (argi < argc && argv[argi][0] != '-')
          configFileName = argv[argi];
        else
          --argi;
        }
      if (configFileName == NULL)
        argerr = 1;
      }
    else if (strncmp(argv[argi], "--config-file=", 14) == 0)
      {
      if (strlen(argv[argi]) > 14)
        configFileName = &argv[argi][14];
      else
        argerr = 1;
      }
    /* Configuration name "-n" or "--config-name" */
    else if (strncmp(argv[argi], "-n", 2) == 0)
      {
      if (strlen(argv[argi]) > 2)
        configName = &argv[argi][2];
      else
        {
        ++argi;
        if (argi < argc && argv[argi][0] != '-')
          configName = argv[argi];
        else
          --argi;
        }
      if (configName == NULL)
        argerr = 1;
      }
    else if (strncmp(argv[argi], "--config-name=", 14) == 0)
      {
      if (strlen(argv[argi]) > 14)
        configName = &argv[argi][14];
      else
        argerr = 1;
      }
    /* Start testcase "-s" or "--start" */
    else if (strncmp(argv[argi], "-s", 2) == 0)
      {
      haveStart = 1;
      if (strlen(argv[argi]) > 2)
        startTestcaseName = &argv[argi][2];
      else
        {
        ++argi;
        if (argi < argc && argv[argi][0] != '-')
          startTestcaseName = argv[argi];
        else
          --argi;
        }
      }
    else if (strncmp(argv[argi], "--start=", 8) == 0)
      {
      haveStart = 1;
      if (strlen(argv[argi]) > 8)
        startTestcaseName = &argv[argi][8];
      else
        argerr = 1;
      }
    else if (strcmp(argv[argi], "--start") == 0)
      haveStart = 1;
    ++argi;
    }
  /* If command line options were incorrect, show usage and terminate */
  if (argerr != 0)
    {
    printf ("Usage: %s [OPTION]...\n", argv[0]);
    printf ("\nOptions:\n");
    printf("  -f, --config-file=FILE    FILE (name or path) to read configurations\n");
    printf("                            with TTNC-3 parameters\n");
    printf("  -n, --config-name=NAME    NAME of configuration to use for initializing\n");
    printf("                            TTCN-3 parameters\n");
    printf("  -s, --start[=TESTCASE]    execute TESTCASE or control part\n\n");
    RTDS_TTCN_RETURN( 0 );
    }
  
  /* If configuration file is present, read its contents */
  if (configFileName != NULL)
    {
    /* If file cannot be opened, error */
    FILE * configFile = fopen(configFileName, "r");
    if (configFile == NULL)
      {
      printf("Error!: %s: cannot open file\n", configFileName);
      RTDS_TTCN_RETURN( 0 );
      }
    /* Read file contents in chunks while decoding each line */
    char * lineString = NULL;
    char bufferString[256];
    size_t lineLength = 0;
    size_t lineMaxLength = 0;
    int linesWithErrors = 0;
    while (fgets(bufferString, sizeof(bufferString), configFile) != NULL)
      {
      /* Read chunks until end of line */
      size_t bufferLength = strlen(bufferString);
      size_t requiredLength = lineLength + bufferLength + 1;
      if (lineMaxLength < requiredLength)
        {
        lineMaxLength = requiredLength;
        lineString = (char *) realloc(lineString, lineMaxLength);
        }
      strcpy(lineString + lineLength, bufferString);
      lineLength += bufferLength;
      /* If end of line, decode */
      if (lineString[lineLength - 1] == '\n')
        {
        ++linesWithErrors;
        /* Remove any CR/LF */
        while (lineLength > 0)
          {
          char * c = &lineString[--lineLength];
          if (*c == '\r' || *c == '\n')
            {
            *c = '\0';
            }
          else
            {
            break;
            }
          }
        /* Reset for reading next line*/
        lineLength = 0;
        /* Parameter type (either MP or TP) */
        char * encdecParameterType = strtok(lineString, ":");
        if (encdecParameterType == NULL)
          {
          continue;
          }
        /* Configuration name */
        char * encdecConfigurationName = strtok(NULL, ":");
        if (encdecConfigurationName == NULL)
          {
          continue;
          }
        /* Module name */
        char * encdecModuleName = strtok(NULL, ":");
        if (encdecModuleName == NULL)
          {
          continue;
          }
        /* Testcase name (if any) */
        char * encdecTestcaseName = NULL;
        if (strcmp(encdecParameterType, "TP") == 0)
          {
          encdecTestcaseName = strtok(NULL, ":");
          if (encdecTestcaseName == NULL)
            {
            continue;
            }
          }
        /* Parameter name */
        char * encdecParameterName = strtok(NULL, ":");
        if (encdecParameterName == NULL)
          {
          continue;
          }
        /* Parameter value */
        char * encdecParameterValue = strtok(NULL, "");
        if (encdecParameterValue == NULL)
          {
          continue;
          }
        /* No errors */
        --linesWithErrors;
        /* Add configuration */
        RTDS_TTCN_EncDecConfiguration * encdecConfiguration = RTDS_TTCN_GetEncDecConfiguration(encdecConfigurationName, TRUE);
        /* Add module */
        RTDS_TTCN_EncDecModule * encdecModule = RTDS_TTCN_GetEncDecModule(encdecConfiguration, encdecModuleName, TRUE);
        /* Add testcase */
        RTDS_TTCN_EncDecTestcase * encdecTestcase = NULL;
        if (encdecTestcaseName != NULL)
          {
          encdecTestcase = RTDS_TTCN_GetEncDecTestcase(encdecModule, encdecTestcaseName, TRUE);
          }
        /* Add parameter */
        RTDS_TTCN_AddEncDecParameter(encdecModule, encdecTestcase, encdecParameterName, encdecParameterValue);
        }
      }
    /* Free allocated memory for chunks (if any) */
    if (lineString != NULL)
      {
      RTDS_FREE(lineString);
      }
    /* If errors were found during decoding, warn user */
    if (linesWithErrors > 0)
      {
      printf("Warning!: %s: incorrect parameter encodings will be ignored\n", configFileName);
      }
    fclose(configFile);
    }
    
  /* Smth should be started automatically */
  if (haveStart)
    {
    RTDS_TTCN_EncDecConfiguration * encdecConfiguration = NULL;
    RTDS_TTCN_EncDecModule * encdecModule = NULL;
    RTDS_TTCN_EncDecTestcase * encdecTestcase = NULL;
    /* If configuration file is required but not present, error */
    if (configFileName == NULL)
      {
      if (!RTDS_TTCN_init_modulepar(NULL))
        {
        printf("Error!: configuration file is missing (use -f or --config-file)\n");
        RTDS_TTCN_ClearEncDecParameters();
        RTDS_TTCN_RETURN( 0 );
        }
      }
    /* If configuration file is present */
    else
      {
      /* If configuration name is not present, error */
      if (configName == NULL)
        {
        printf("Error!: configuration name is missing (use -n or --config-name)\n");
        RTDS_TTCN_ClearEncDecParameters();
        RTDS_TTCN_RETURN( 0 );
        }
      /* If configuration is not found, error */
      encdecConfiguration = RTDS_TTCN_GetEncDecConfiguration(configName, FALSE);
      if (encdecConfiguration == NULL)
        {
        printf("Error!: %s: configuration not found\n", configName);
        RTDS_TTCN_ClearEncDecParameters();
        RTDS_TTCN_RETURN( 0 );
        }
      /* If errors occured while decoding parameters, done */
      if (!RTDS_TTCN_init_modulepar(encdecConfiguration))
        {
        RTDS_TTCN_ClearEncDecParameters();
        RTDS_TTCN_RETURN( 0 );
        }
      }
    /* Use root module */
    strcpy(currentModuleName, RTDS_TTCN_defaultRootModule);
    currentModuleId.moduleName = currentModuleName;
    currentModuleId.baseName = currentModuleName;
    tciRootModule(&currentModuleId);
    /* If testcase is present, execute it */
    if (startTestcaseName != NULL)
      {
      testcasesList = tciGetTestCases();
      if (testcasesList != NULL)
        {
        for(testcaseIndex = 0; testcaseIndex < testcasesList->length; ++testcaseIndex)
          {
          if (strcmp(testcasesList->testcaseList[testcaseIndex].baseName, startTestcaseName) == 0)
            {
            encdecModule = RTDS_TTCN_GetEncDecModule(encdecConfiguration, testcasesList->testcaseList[testcaseIndex].moduleName, FALSE);
            encdecTestcase = RTDS_TTCN_GetEncDecTestcase(encdecModule, startTestcaseName, FALSE);
            break;
            }
          }
        }
      int r = executeTestcase(startTestcaseName, encdecTestcase);
      RTDS_TTCN_ClearEncDecParameters();
      RTDS_TTCN_RETURN( r );
      }
    /* If testcase is not present, execute control part */
    else
      {
      int r = executeControlPart();
      RTDS_TTCN_ClearEncDecParameters();
      RTDS_TTCN_RETURN( r );
      }
    }
  
  /* Entering interactive mode... */
  
  /* If configuration file is required but not present, error */
  if (configFileName == NULL && !RTDS_TTCN_init_modulepar(NULL))
    {
    printf("Error!: configuration file is missing (use -f or --config-file)\n");
    RTDS_TTCN_ClearEncDecParameters();
    RTDS_TTCN_RETURN( 0 );
    }
  
  /* Use root module */
  strcpy(currentModuleName, RTDS_TTCN_defaultRootModule);
  currentModuleId.moduleName = currentModuleName;
  currentModuleId.baseName = currentModuleName;
  tciRootModule(&currentModuleId);
  
  /* Display banner */
  printf("========================================================\n");
  printf("=                   P R A G M A D E V                  =\n");
  printf("=                      S T U D I O                     =\n");
  printf("=------------------------------------------------------=\n");
  printf("= TTCN-3 test control interactive command line utility =\n");
  printf("========================================================\n");
    
  unsigned char                   loopLevel           = 1;    /* Levels: 1 = configurations; 2 = modules; 3 = testcases */
  RTDS_TTCN_EncDecConfiguration * encdecConfiguration = NULL;
  RTDS_TTCN_EncDecModule        * encdecModule        = NULL;
  RTDS_TTCN_EncDecTestcase      * encdecTestcase      = NULL;
  /* Interactive loop */
  for ( ; ; )
    {
    /* Loop configurations (level 1) */
    while (loopLevel == 1)
      {
      encdecConfiguration = NULL;
      /* If there are no configurations, nothing to do here, jump to modules */
      if (RTDS_TTCN_encdecConfigurations == NULL)
        {
        loopLevel = 2;
        break;
        }
      /* If configuration name is present, use it and jump to modules */
      if (configName != NULL)
        {
        encdecConfiguration = RTDS_TTCN_GetEncDecConfiguration(configName, FALSE);
        if (encdecConfiguration == NULL)
          {
          printf("Error!: %s: configuration not found\n", configName);
          RTDS_TTCN_ClearEncDecParameters();
          RTDS_TTCN_RETURN( 0 );
          }
        configName = NULL;
        loopLevel = 2;
        }
      /* If configuration name is not present, show available configuration to chose from */
      if (encdecConfiguration == NULL)
        {
        printf("\nAvailable parameter configurations:\n");
        /* Sort configurations by name */
        sortedNames = NULL;
        for (encdecConfiguration = RTDS_TTCN_encdecConfigurations; encdecConfiguration != NULL; encdecConfiguration = encdecConfiguration->next)
          {
          sortedNames = sortedInsert(sortedNames, encdecConfiguration->name);
          }
        /* List configurations */
        configurationIndex = 0;
        for (sortedNamesElement = sortedNames; sortedNamesElement != NULL; sortedNamesElement = sortedNamesElement->next)
          {
          ++configurationIndex;
          sortedNamesElement->chosenIndex = configurationIndex;
          printf("%4d. %s\n", configurationIndex, sortedNamesElement->string);
          }
        /* Get user input */
        printf("\nEnter number - <return> = go back - \"q\" = quit: ");
        /* If smth went wrong, exit */
        if (fgets(chosenOption, sizeof(chosenOption), stdin) == NULL)
          {
          printf("\n");
          freeStringLinkedList(sortedNames);
          RTDS_TTCN_ClearEncDecParameters();
          RTDS_TTCN_RETURN( 0 );
          }
        /* On <return> re-display configurations */
        if (chosenOption[0] == '\n')
          {
          freeStringLinkedList(sortedNames);
          continue;
          }
        /* On "q" quit */
        if (strcmp(chosenOption, "q\n") == 0)
          {
          freeStringLinkedList(sortedNames);
          RTDS_TTCN_ClearEncDecParameters();
          RTDS_TTCN_RETURN( 0 );
          }
        /* On number input, check it is a valid choice */
        chosenIndex = (int) strtol(chosenOption, NULL, 10);
        if (chosenIndex < 1 || chosenIndex > configurationIndex)
          {
          freeStringLinkedList(sortedNames);
          continue;
          }
        /* Remember chosen configuration, and jump to modules on the next iteration */
        for (sortedNamesElement = sortedNames; sortedNamesElement != NULL && sortedNamesElement->chosenIndex != chosenIndex; sortedNamesElement = sortedNamesElement->next);
        encdecConfiguration = RTDS_TTCN_GetEncDecConfiguration(sortedNamesElement->string, FALSE);
        freeStringLinkedList(sortedNames);
        loopLevel = 2;
        }
      /* If parameters initialization fails, exit */
      if (!RTDS_TTCN_init_modulepar(encdecConfiguration))
        {
        RTDS_TTCN_ClearEncDecParameters();
        RTDS_TTCN_RETURN( 0 );
        }
      } /* End loop configurations */
      
    /* Loop modules (level 2) */
    while (loopLevel == 2)
      {
      printf("\nAvailable modules:\n");
      /* Sort modules by name */
      moduleList = RTDS_TTCN_GetAllRootModules();
      sortedNames = NULL;
      for (moduleIndex = 0; moduleIndex < moduleList->length; ++moduleIndex)
        {
        sortedNames = sortedInsert(sortedNames, moduleList->modList[moduleIndex].moduleName);
        }
      moduleIndex = 0;
      sortedNamesElement = sortedNames;
      while (sortedNamesElement != NULL)
        {
        /* Figure out if module has a control part and/or testcases */
        tempModuleId.moduleName = sortedNamesElement->string;
        tempModuleId.baseName = sortedNamesElement->string;
        hasControl = (RTDS_TTCN_GetProcessNumber(&tempModuleId) != 0);
        tciRootModule(&tempModuleId);
        testcasesList = tciGetTestCases();
        nbTestcases = 0;
        if (testcasesList != NULL)
          {
          for (testcaseIndex = 0; testcaseIndex < testcasesList->length; ++testcaseIndex)
            {
            if (strcmp(testcasesList->testcaseList[testcaseIndex].moduleName, sortedNamesElement->string) == 0)
              {
              ++nbTestcases;
              }
            }
          }
        /* If module has a control part or testcases, print it */
        if (hasControl || nbTestcases != 0)
          {
          ++moduleIndex;
          sortedNamesElement->chosenIndex = moduleIndex;
          printf("%4d. %s (", moduleIndex, sortedNamesElement->string);
          if (hasControl)
            {
            printf("control part");
            if (nbTestcases != 0)
              {
              printf(" & ");
              }
            }
          if (nbTestcases != 0)
            {
            printf("%d testcase%s", nbTestcases, nbTestcases == 1 ? "" : "s");
            }
          printf(")\n");
          }
        sortedNamesElement = sortedNamesElement->next;
        }
      /* Get user input */
      printf("\nEnter number - <return> = go back - \"q\" = quit: ");
      /* If smth went wrong, exit */
      if (fgets(chosenOption, sizeof(chosenOption), stdin) == NULL)
        {
        printf("\n");
        freeStringLinkedList(sortedNames);
        RTDS_TTCN_ClearEncDecParameters();
        RTDS_TTCN_RETURN( 0 );
        }
      /* On <return> go back to re-display configurations */
      if (chosenOption[0] == '\n')
        {
        freeStringLinkedList(sortedNames);
        loopLevel = 1;
        break;
        }
      /* On "q" quit */
      if (strcmp(chosenOption, "q\n") == 0)
        {
        freeStringLinkedList(sortedNames);
        RTDS_TTCN_ClearEncDecParameters();
        RTDS_TTCN_RETURN( 0 );
        }
      /* On number input, check it is a valid choice */
      chosenIndex = (int) strtol(chosenOption, NULL, 10);
      if (chosenIndex < 1 || chosenIndex > moduleIndex)
        {
        freeStringLinkedList(sortedNames);
        continue;
        }
      /* Set chosen module as current, and jump to testcases on the next iteration */
      for (sortedNamesElement = sortedNames; sortedNamesElement != NULL && sortedNamesElement->chosenIndex != chosenIndex; sortedNamesElement = sortedNamesElement->next);
      strcpy(currentModuleName, sortedNamesElement->string);
      tciRootModule(&currentModuleId);
      freeStringLinkedList(sortedNames);
      loopLevel = 3;
    } /* End loop modules*/


    /* Loop testcases (level 3) */
    while (loopLevel == 3)
      {
      testcasesList = tciGetTestCases();
      if (testcasesList == NULL)
        {
        loopLevel = 2;
        break;
        }
      printf("\nAvailable testcases (or control part):\n");
      sortedNames = NULL;
      for(testcaseIndex = 0; testcaseIndex < testcasesList->length; ++testcaseIndex)
        {
        if (strcmp(testcasesList->testcaseList[testcaseIndex].moduleName, currentModuleName) == 0)
			    {
				  sortedNames = sortedInsert(sortedNames, testcasesList->testcaseList[testcaseIndex].baseName);
			    }
        }
      testcaseIndex = 0;
      if (RTDS_TTCN_GetProcessNumber(&currentModuleId) != 0)
        {
        ++testcaseIndex;
        printf("%4d. control\n", testcaseIndex);
        }
      sortedNamesElement = sortedNames;
	    while (sortedNamesElement != NULL)
	      {
	      ++testcaseIndex;
	      sortedNamesElement->chosenIndex = testcaseIndex;
	      printf("%4d. %s\n", testcaseIndex, sortedNamesElement->string);
	      sortedNamesElement = sortedNamesElement->next;
	      }
      /* Input testcase to run */
      for ( ; ; )
        {
        printf("\nEnter number - <return> = go back - \"q\" = quit: ");
        if (fgets(chosenOption, sizeof(chosenOption), stdin) == NULL)
          {
          printf("\n");
          freeStringLinkedList(sortedNames);
          RTDS_TTCN_ClearEncDecParameters();
          RTDS_TTCN_RETURN( 0 );
          }
        if (chosenOption[0] == '\n')
          {
          loopLevel = 2;
          break;
          }
        if (strcmp(chosenOption, "q\n") == 0)
          {
          freeStringLinkedList(sortedNames);
          RTDS_TTCN_ClearEncDecParameters();
          RTDS_TTCN_RETURN( 0 );
          }
        chosenIndex = (int) strtol(chosenOption, NULL, 10);
        if (chosenIndex < 1 || chosenIndex > testcaseIndex)
          {
          continue;
          }
        if (chosenIndex == 1 && RTDS_TTCN_GetProcessNumber(&currentModuleId) != 0)
          {
          printf("\n================ CONTROL PART EXECUTION ================\n");
	        finalVerdict = executeControlPart();
          }
        else
          {
          for (sortedNamesElement = sortedNames; sortedNamesElement != NULL && sortedNamesElement->chosenIndex != chosenIndex; sortedNamesElement = sortedNamesElement->next);
          printf("\n================== TESTCASE EXECUTION ==================\n");
          encdecModule = RTDS_TTCN_GetEncDecModule(encdecConfiguration, currentModuleName, FALSE);
          encdecTestcase = RTDS_TTCN_GetEncDecTestcase(encdecModule, sortedNamesElement->string, FALSE);
	        finalVerdict = executeTestcase(sortedNamesElement->string, encdecTestcase);
          }
        if (finalVerdict != TCI_ERROR)
          {
          printf("Final verdict: %s", verdict_string[finalVerdict]);
          }
	      printf("\n==================== EXECUTION END =====================\n");
	      }
	    freeStringLinkedList(sortedNames);
	    } /* End loop testcases */
    } /* End interactive loop */
  RTDS_TTCN_ClearEncDecParameters();
  RTDS_TTCN_RETURN( 0 );
  }

