#ifndef _RTDS_BASIC_TYPES_H_
#define _RTDS_BASIC_TYPES_H_

#include <winsock2.h>
#include <windows.h>

#define RTDS_THREAD_START_FUNC LPTHREAD_START_ROUTINE

/**********************************
   System defines: OK/ERROR/NONE
 **********************************/
#define RTDS_OK                 1
#define RTDS_ERROR              ERROR   /* ERROR is defined in the windows library */
#define RTDS_TIMEOUT            2       /* timeout return value (ususally for semaphores */
#define NONE                    0       /* default option to send */
#define NOT_AVAILABLE           0       /* unavailable options default value */


/****************************************
    Define for RTDS_OS and RDTS_MACRO
 ****************************************/
#define NO_WAIT                         NONE                    /* Not used */
#define MSG_PRI_NORMAL                  NONE                    /* Not used */
#define RTDS_SEMAPHORE_MAX_NAME_LENGTH  20                      /* Maximum length of a semaphore name: No restriction for windows */
#define RTDS_TASK_OPTIONS               NONE                    /* taskSpawn options */
#define RTDS_TASK_STACK_SIZE            5000                    /* Stack size when creating a new task */
#define RTDS_TIMER_TASK_STACK_SIZE      50                      /* Stack size for the timer threads */
#define RTDS_SEMAPHORE_MAXCOUNT         1000                    /* Max counter value for counting semaphore */



/********************************************************
 * Define for SDL Symbols (generated by RTDS generator) *
 ********************************************************/
#define RTDS_SEMAPHORE_TIME_OUT_FOREVER     INFINITE        /* Time out option when taking the semaphore : wait forever */
#define RTDS_SEMAPHORE_TIME_OUT_NO_WAIT     0               /* Time out option when taking the semaphore : no wait */
#define RTDS_SEMAPHORE_OPTION_PRIO          NOT_AVAILABLE   /* Semaphore creation option: queuing based on priority */
#define RTDS_SEMAPHORE_OPTION_FIFO          NOT_AVAILABLE   /* Semaphore creation option: queuing based on FIFO */
#define RTDS_BINARY_SEM_INITIAL_EMPTY       0               /* Binary semaphore creation initial state: empty */
#define RTDS_BINARY_SEM_INITIAL_FULL        1               /* Binary semaphore creation initial state: full */
#define RTDS_MUTEX_SEM_DELETE_SAFE          NOT_AVAILABLE   /* Not used: Protects a task that owns the semaphore from unexpected deletion */
#define RTDS_MUTEX_SEM_INVERSION_SAFE       NOT_AVAILABLE   /* Not used: Protect the system from priority inversion */


/*******************************************************************
 *                  Default priorities
 *******************************************************************
 * Possible values for windows thread priority (Winbase.h)
 *      Const THREAD_PRIORITY_IDLE          = -15
 *      Const THREAD_PRIORITY_LOWEST        =  -2
 *      Const THREAD_PRIORITY_BELOW_NORMAL  =  -1
 *      Const THREAD_PRIORITY_NORMAL        =   0
 *      Const THREAD_PRIORITY_ABOVE_NORMAL  =   1
 *      Const THREAD_PRIORITY_HIGHEST       =   2
 *      Const THREAD_PRIORITY_TIME_CRITICAL =  15
 *******************************************************************/
#define RTDS_DEFAULT_PROCESS_PRIORITY       THREAD_PRIORITY_NORMAL
#define RTDS_DEFAULT_TCP_CLIENT_PRIORITY    THREAD_PRIORITY_TIME_CRITICAL
#define RTDS_TIMER_PRIORITY                 THREAD_PRIORITY_TIME_CRITICAL



/******************************************************************
 *                      Semaphore Information
 * ----------------------------------------------------------------
 * SDL-RT semaphores use mutex and counting Windows semaphores
 ******************************************************************/
enum RTDS_SemaphoreType { COUNTING, BINARY, MUTEX };

typedef struct RTDS_SemaphoreInfo
    {
    HANDLE                  semId;
    enum RTDS_SemaphoreType semType;
    } RTDS_SemaphoreInfo;

/*
 * TYPE RTDS_SEMAPHORE.ID:
 * ----------------------
 * Type for a semaphore identifier
 */
typedef RTDS_SemaphoreInfo * RTDS_SemaphoreId;


typedef void * FUNCPTR;


/******************************************************************
 *        SEMAPHORE FUNCTIONS and TYPE DECLARATIONS
 *******************************************************************/
extern RTDS_SemaphoreId RTDS_SemMCreate ( void );
extern RTDS_SemaphoreId RTDS_SemCCreate ( int opt, int initial_state );
extern RTDS_SemaphoreId RTDS_SemBCreate ( int initial_state );
extern void             RTDS_SemDelete  ( RTDS_SemaphoreId SemId_To_Delete );
extern void             RTDS_SemGive    ( RTDS_SemaphoreId SemId_To_Give );
extern int              RTDS_SemTake    ( RTDS_SemaphoreId SemId_To_Give, long time_to_Wait );


/******************************************************************
 *            THREAD BASED TIMER FUNCTIONS
 *******************************************************************/
extern int      RTDS_TimerThreadDelete      ( HANDLE timerThreadId );
extern DWORD    RTDS_TimerThreadFunction    ( void * params );

/*
 * TYPE RTDS_PROCESS.ID:
 * ----------------------
 * Type for process identifier
 */
typedef DWORD RTDS_RtosTaskId;

/*
 * TYPE RTDS_PROCESS.PRIORITY.
 * ----------------------
 * Type for process priority
 */
typedef int RTDS_ProcessPriority;

/*
 * TYPE RTDS_SEMAPHORE_STATUS:
 * ----------------------
 * Type for a semaphore status
 */
typedef int RTDS_SemaphoreStatus;

/*
 * TYPE RTDS_SEMAPHORE_TIMEOUT:
 * ----------------------
 * Type for a semaphore timeout value
 */
typedef long RTDS_SemaphoreTimeout;


/******************************************************************
 *        STRUCT RTDS_QCB : Queue Control Block
 *******************************************************************/
typedef struct RTDS_QCB
    {
    RTDS_SemaphoreId            chainedListSemId;  /* Chained list semaphore Id */
    RTDS_SemaphoreId            blockingSemId;     /* Blocking semaphore Id */
    struct RTDS_MessageHeader   * queue;           /* Pointer on the first message in the queue */
    } RTDS_QCB;


/*
 * TYPE RTDS_QUEUE.Id:
 * ----------------------
 * Type for a message queue identifier
 */
typedef RTDS_QCB * RTDS_RtosQueueId;

/*
 * STRUCT RTDS_TIMER.STATE:
 * ------------------------
 * Element in chained list for all active timers
 */

typedef struct RTDS_TimerState
    {
    long                        timerNumber;        /* Message number for the timer name                           */
    long                        timerUniqueId;      /* Timer's unique identifier among all timers                  */
    unsigned long               timeoutValue;       /* System tick counter value when it will go off               */
    struct RTDS_SdlInstanceId   * receiverId;       /* The message queue of the receiver                           */
    int                         state;              /* State for the timer (RTDS_TIMER_OK or RTDS_TIMER_CANCELLED) */
    HANDLE                      watchDogId;         /* Id of the timer thread                                      */
    struct RTDS_TimerState      * next;             /* 0x18 Next timer in list                                     */
    } RTDS_TimerState;

/*
 * STRUCT RTDS_GLOBAL.SEMAPHORE.INFO:
 * -------------------------------
 * Type for the list of semaphores in the system with their information
 */
typedef struct RTDS_GlobalSemaphoreInfo
    {
    RTDS_SemaphoreId                semaphoreId;        /* The semaphore address    */
    int                             semaphoreNumber;    /* Semaphore number */
    struct RTDS_GlobalSemaphoreInfo * next;             /* Next semaphoreInfo */
    } RTDS_GlobalSemaphoreInfo;

/*
 * Additional fields in common types
 * ---------------------------------
 */
/* Message header: none */
#define RTDS_MESSAGE_HEADER_ADDITIONNAL_FIELDS
/* Process information: add priority */
#define RTDS_GLOBAL_PROCESS_INFO_ADDITIONNAL_FIELDS \
  RTDS_ProcessPriority priority;

#endif /* _RTDS_BASIC_TYPES_H_ */

