/* -*- Mode:C++; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Fabian Mauchle <fabian.mauchle@hsr.ch>
 */

#include "ns3/test.h"
#include "ipv6-extension-header.h"
#include "ipv6-option-header.h"

using namespace ns3;

// ===========================================================================
// An empty option field must be filled with pad1 or padN header so the
// extension header's size is a multiple of 8.
//
// 0                                                              31
// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
// |  Extension Destination Header |                               |
// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+          PadN Header          +
// |                                                               |
// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
// ===========================================================================
class TestEmptyOptionField : public TestCase
{
public:
  TestEmptyOptionField () : TestCase("TestEmptyOptionField") {}

  virtual bool DoRun()
  {
    Ipv6ExtensionDestinationHeader header;
    NS_TEST_EXPECT_MSG_EQ(header.GetSerializedSize() % 8, 0, "length of extension header is not a multiple of 8");

    Buffer buf;
    buf.AddAtStart(header.GetSerializedSize());
    header.Serialize(buf.Begin());

    const uint8_t* data = buf.PeekData();
    NS_TEST_EXPECT_MSG_EQ(*(data+2), 1, "padding is missing"); //expecting a padN header

    return GetErrorStatus();
  }
};

// ===========================================================================
// An option without alignment requirement must not be padded
//
// 0                                                              31
// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
// |  Extension Destination Header | OptionWithoutAlignmentHeader..|
// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
// |..OptionWithoutAlignmentHeader |          PadN Header          |
// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
// ===========================================================================
class OptionWithoutAlignmentHeader : public Ipv6OptionHeader
{
public:
  static const uint8_t TYPE = 42;
  virtual uint32_t GetSerializedSize () const
  {
    return 4;
  }

  virtual void Serialize (Buffer::Iterator start) const
  {
    start.WriteU8(TYPE);
    start.WriteU8(GetSerializedSize()-2);
    start.WriteU16(0);
  }
};


class TestOptionWithoutAlignment : public TestCase
{
public:
  TestOptionWithoutAlignment () : TestCase("TestOptionWithoutAlignment") {}

  virtual bool DoRun()
  {
    Ipv6ExtensionDestinationHeader header;
    OptionWithoutAlignmentHeader optionHeader;
    header.AddOption(optionHeader);


    NS_TEST_EXPECT_MSG_EQ(header.GetSerializedSize() % 8, 0, "length of extension header is not a multiple of 8");

    Buffer buf;
    buf.AddAtStart(header.GetSerializedSize());
    header.Serialize(buf.Begin());

    const uint8_t* data = buf.PeekData();
    NS_TEST_EXPECT_MSG_EQ(*(data+2), OptionWithoutAlignmentHeader::TYPE, "option without alignment is not first in header field");

    return GetErrorStatus();
  }
};

// ===========================================================================
// An option with alignment requirement must be padded accordingly (padding to
// a total size multiple of 8 is allowed)
//
// 0                                                              31
// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
// |  Extension Destination Header |          PadN Header          |
// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
// |                   OptionWithAlignmentHeader                   |
// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
// |          PadN Header          |                               |
// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+                               +
// |                   Ipv6OptionJumbogramHeader                   |
// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
// ===========================================================================
class OptionWithAlignmentHeader : public Ipv6OptionHeader
{
public:
  static const uint8_t TYPE = 73;
  virtual uint32_t GetSerializedSize () const
  {
    return 4;
  }

  virtual void Serialize (Buffer::Iterator start) const
  {
    start.WriteU8(TYPE);
    start.WriteU8(GetSerializedSize()-2);
    start.WriteU16(0);
  }

  virtual Alignment GetAlignment() const
  {
    return (Alignment){4,0};
  }
};


class TestOptionWithAlignment : public TestCase
{
public:
  TestOptionWithAlignment () : TestCase("TestOptionWithAlignment") {}

  virtual bool DoRun()
  {
    Ipv6ExtensionDestinationHeader header;
    OptionWithAlignmentHeader optionHeader;
    header.AddOption(optionHeader);
    Ipv6OptionJumbogramHeader jumboHeader; //has an alignment of 4n+2
    header.AddOption(jumboHeader);

    NS_TEST_EXPECT_MSG_EQ(header.GetSerializedSize() % 8, 0, "length of extension header is not a multiple of 8");

    Buffer buf;
    buf.AddAtStart(header.GetSerializedSize());
    header.Serialize(buf.Begin());

    const uint8_t* data = buf.PeekData();
    NS_TEST_EXPECT_MSG_EQ(*(data+2), 1, "padding is missing"); //expecting a padN header
    NS_TEST_EXPECT_MSG_EQ(*(data+4), OptionWithAlignmentHeader::TYPE, "option with alignment is not padded correctly");
    NS_TEST_EXPECT_MSG_EQ(*(data+8), 1, "padding is missing"); //expecting a padN header
    NS_TEST_EXPECT_MSG_EQ(*(data+10), jumboHeader.GetType(), "option with alignment is not padded correctly");

    return GetErrorStatus();
  }
};

// ===========================================================================
// An option with an alignment that exactly matches the gap must not be padded
// (padding to a total size multiple of 8 is allowed)
//
// 0                                                              31
// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
// |  Extension Destination Header |                               |
// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+                               +
// |                   Ipv6OptionJumbogramHeader                   |
// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
// |                   OptionWithAlignmentHeader                   |
// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
// |                           PadN Header                         |
// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
// ===========================================================================

class TestFulfilledAlignment : public TestCase
{
public:
  TestFulfilledAlignment () : TestCase("TestCorrectAlignment") {}

  virtual bool DoRun()
  {
    Ipv6ExtensionDestinationHeader header;
    Ipv6OptionJumbogramHeader jumboHeader; //has an alignment of 4n+2
    header.AddOption(jumboHeader);
    OptionWithAlignmentHeader optionHeader;
    header.AddOption(optionHeader);

    NS_TEST_EXPECT_MSG_EQ(header.GetSerializedSize() % 8, 0, "length of extension header is not a multiple of 8");

    Buffer buf;
    buf.AddAtStart(header.GetSerializedSize());
    header.Serialize(buf.Begin());

    const uint8_t* data = buf.PeekData();
    NS_TEST_EXPECT_MSG_EQ(*(data+2), jumboHeader.GetType(), "option with fulfilled alignment is padded anyway");
    NS_TEST_EXPECT_MSG_EQ(*(data+8), OptionWithAlignmentHeader::TYPE, "option with fulfilled alignment is padded anyway");

    return GetErrorStatus();
  }
};

class Ipv6ExtensionHeaderTestSuite : public TestSuite
{
public:
  Ipv6ExtensionHeaderTestSuite ()
    : TestSuite ("ipv6-extension-header", UNIT)
  {
    AddTestCase (new TestEmptyOptionField);
    AddTestCase (new TestOptionWithoutAlignment);
    AddTestCase (new TestOptionWithAlignment);
    AddTestCase (new TestFulfilledAlignment);

  }
};

Ipv6ExtensionHeaderTestSuite ipv6ExtensionHeaderTestSuite;
